<?php
/**
 * NOTICE OF LICENSE
 *
 * This source file is subject to a commercial license from Common-Services Co., Ltd.
 * Use, copy, modification or distribution of this source file without written
 * license agreement from the SARL SMC is strictly forbidden.
 * In order to obtain a license, please contact us: contact@common-services.com
 * ...........................................................................
 * INFORMATION SUR LA LICENCE D'UTILISATION
 *
 * L'utilisation de ce fichier source est soumise a une licence commerciale
 * concedee par la societe Common-Services Co., Ltd.
 * Toute utilisation, reproduction, modification ou distribution du present
 * fichier source sans contrat de licence ecrit de la part de la Common-Services Co. Ltd. est
 * expressement interdite.
 * Pour obtenir une licence, veuillez contacter Common-Services Co., Ltd. a l'adresse: contact@common-services.com
 *
 * @author    Olivier B.
 * @copyright Copyright (c) 2011-2018 Common Services Co Ltd - 90/25 Sukhumvit 81 - 10260 Bangkok - Thailand
 * @license   Commercial license
 * @package   Amazon Market Place
 * Support by mail:  support.amazon@common-services.com
*/

require_once(dirname(__FILE__).'/env.php');
require_once(dirname(__FILE__).'/../amazon.php');

require_once(dirname(__FILE__).'/../classes/amazon.context.class.php');
require_once(dirname(__FILE__).'/../classes/amazon.product.class.php');
require_once(dirname(__FILE__).'/../classes/amazon.tools.class.php');
require_once(dirname(__FILE__).'/../classes/amazon.batch.class.php');
require_once(dirname(__FILE__).'/../classes/amazon.address.class.php');
require_once(dirname(__FILE__).'/../classes/amazon.order_info.class.php');
require_once(dirname(__FILE__).'/../classes/amazon.order.class.php');
require_once(dirname(__FILE__).'/../classes/amazon.orderhistory.class.php');
require_once(dirname(__FILE__).'/../classes/amazon.cart.class.php');
require_once(dirname(__FILE__).'/../classes/amazon.payment.class.php');
require_once(dirname(__FILE__).'/../classes/amazon.webservice.class.php');
require_once(dirname(__FILE__).'/../classes/amazon.orders_reports.class.php');

class AmazonListOrder extends Amazon
{
    public static $errors     = array();
    public static $warnings   = array();
    public static $messages   = array();
    public static $orders     = array();
    private $_amazonApi = null;
    private $marketplace_iso2id = null;
    private $marketplace_region = null;
    private $marketplace_lang2region = null;
    private $marketplace_id_lang = null;
    private $marketplace_setup = null;
    private $orders_reports_management = false;
    private $_debug     = false;

    public function __construct()
    {
        parent::__construct();

        AmazonContext::restore($this->context);

        $this->amazon_features = $this->getAmazonFeatures();
    }

    public static function jsonDisplayExit()
    {
        if ($result = ob_get_clean()) {
            $output = $result;
        } else {
            $output = null;
        }

        $json = Tools::jsonEncode(array(
            'orders' => AmazonListOrder::$orders,
            'count' => count(AmazonListOrder::$orders),
            'error' => (count(AmazonListOrder::$errors) ? true : false),
            'errors' => AmazonListOrder::$errors,
            'output' => $output
        ));
        if ($callback = Tools::getValue('callback')) {
            // jquery

            echo (string)$callback.'('.$json.')';
            die;
        } else {
            // cron

            echo $json;
            die;
        }
    }

    public function dispatch($action)
    {
        $this->_debug = (bool)Configuration::get('AMAZON_DEBUG_MODE');

        // Merge Order API results & Reports results
        //
        $this->orders_reports_management = (bool)Configuration::get('AMAZON_ORDERS_REPORTS', false);

        if (Tools::getValue('debug')) {
            $this->_debug = true;
        }

        if ($this->_debug) {
            @ini_set('display_errors', 'on');

            @error_reporting(E_ALL | E_STRICT);

            // Generate a warning to identify which program/module sent the headers previously
            if (Tools::getValue('debug_header') !== false) {
                echo ob_get_clean();
                header('Pragma: no-cache');
            }
        }
        $this->init();

        switch ($action) {
            case 'list':
                $this->displayList();
                break;
            case 'report':
                if ($this->orders_reports_management) {
                    $this->manageOrdersReport();
                }
                break;
            
        }
    }


    protected function manageOrdersReport()
    {
        $no_pending_order_string = $this->l('No pending order for the selected period');
        $start_time = time();
        $report_id = Tools::getValue('report-id');
        $region = $this->marketplace_setup['params']['Country'];

        // Listing Reports
        //
        $this->amazon_report_ws = new AmazonOrdersReports($this->marketplace_setup['auth'], $this->marketplace_setup['params'], null, Amazon::$debug_mode);

        if (Amazon::$debug_mode) {
            echo '<pre>'.Amazon::LF;
            printf('%s(#%d): - Order Report Request:'.Amazon::LF, basename(__FILE__), __LINE__);
            echo '</pre>'.Amazon::LF;
        }
        $batches = new AmazonBatches('batch_order_report');
        $last_import = $batches->getLastForRegion($region);
        $batch = null;

        if ($last_import) {
            $batch = $batches->getCurrent();
            $last_import = date('c', strtotime($last_import));
        } else {
            $last_import = date('c', strtotime('yesterday'));
        }

        $request_list = $this->amazon_report_ws->reportList($last_import);

        if (Amazon::$debug_mode) {
            echo '<pre>'.Amazon::LF;
            printf('%s(#%d): - Order Report Request List: %s'.Amazon::LF, basename(__FILE__), __LINE__, print_r($request_list, true));
            echo '</pre>'.Amazon::LF;
        }

        if (is_array($request_list) && count($request_list)) {
            $report_id = null;
            $requested_report = reset($request_list);

            switch ($requested_report->ReportType) {
                case '_GET_FLAT_FILE_ORDERS_DATA_':
                    $report_id = (string)$requested_report->ReportId;
                    break;
                default:
                    printf('%s', $this->l('No available reports'));
                    break;
            }

            if (!$report_id) {
                if (Amazon::$debug_mode) {
                    echo '<pre>'.Amazon::LF;
                    printf('%s(#%d): - Order Report: Report not yet available'.Amazon::LF, basename(__FILE__), __LINE__);
                    echo '</pre>'.Amazon::LF;
                }
                die;
            }

            $report_data = $this->amazon_report_ws->getReport($report_id);

            if (Tools::strlen($report_data)) {
                $this->processReport($report_data);

                $this->amazon_report_ws->ackReport($report_id);
            }
        } elseif (strtotime($last_import) < (time() - AmazonOrdersReports::LAG)) {
            // Request a report for the period
            //
            if (!($report_request_id = $this->amazon_report_ws->reportRequest($last_import))) {
                AmazonListOrder::$errors[] = sprintf('%s (%d/%s)', $this->l('Failed to request a report'), __LINE__, 'Report Request');
                die;
            } else {
                $batches = new AmazonBatches('batch_order_report');
                $batch = new AmazonBatch($start_time);
                $batch->id = $report_request_id;
                $batch->timestop = time();
                $batch->type = 'Report (Orders)';
                $batch->region = $region;
                $batch->created = 0;
                $batch->updated = 0;
                $batch->deleted = 0;
                $batches->add($batch);
                $batches->save();

                printf('%s... (%d)', $this->l('An order report has been requested, please wait a while'), $report_request_id);
            }
        }
    }


    public function processReport($result)
    {
        if ($result == null or empty($result)) {
            AmazonListOrder::$errors[] = sprintf('%s(#%d): processReport - Report is empty !', basename(__FILE__), __LINE__);
            return (false);
        }
        
        if (strstr($result, '<Error>') !== false) {
            AmazonListOrder::$errors[] = sprintf('%s(#%d): processReport - Error: %s', basename(__FILE__), __LINE__, print_r($result));
            return (false);
        }
        
        $lines = explode("\n", utf8_encode($result));

        if (!is_array($lines) || !count($lines)) {
            AmazonListOrder::$errors[] = sprintf('%s(#%d): processReport - Report is empty !', basename(__FILE__), __LINE__);
            return (false);
        }

        if (Amazon::$debug_mode) {
            echo "<pre>\n";
            echo str_repeat('-', 160)."\n";
            printf('Orders: %s lines'."<br />\n", count($lines));
            echo "</pre>\n";
        }

        $header = reset($lines);

        if (Amazon::$debug_mode) {
            echo "<pre>\n";
            echo str_repeat('-', 160)."\n";
            printf('Header: %s'."<br />\n", nl2br(print_r(explode("\t", $header), true)));
            echo "</pre>\n";
        }

        $amazon_orders = array();
        $count = 0;
        $orders_fields_count = count(AmazonAddress::$orders_fields);

        foreach ($lines as $line) {
            if (empty($line)) {
                continue;
            }
            if ($count++ < 1) {
                continue;
            }

            $order = array();
            $result = explode("\t", $line);

            if (count(array_keys($result)) < $orders_fields_count) {
                continue;
            }
            $mp_order_id = $result[0];

            foreach (AmazonAddress::$orders_fields as $key => $order_field) {
                if (isset(AmazonAddress::$report_mapping[$order_field])) {
                    $order[$order_field] = $result[$key];
                }
            }
            $amazon_orders[$mp_order_id] = $order;

            echo "<pre>\n";
            echo str_repeat('-', 160)."\n";
            printf('Order: %s'."<br />\n", print_r($order, true));
            echo "</pre>\n";
        }

        if (is_array($amazon_orders) && count($amazon_orders)) {
            // Cosmetic
            $mappings = array();
            $db_values = array();

            foreach (AmazonAddress::$report_mapping as $key => $mapping) {
                $mappings[$key] = pSQL($mapping);
            }
            $fields = implode('`, `', $mappings);

            foreach ($amazon_orders as $mp_order_id => $order) {
                foreach ($order as $key => $value) {
                    $db_values[$key] = pSQL($value);
                }
                $values = implode('", "', $db_values);
                $sql = sprintf('REPLACE INTO `'._DB_PREFIX_.Amazon::TABLE_MARKETPLACE_ORDER_ADDRESS.'` (`%s`) values("%s");', $fields, $values);

                if (Amazon::$debug_mode) {
                    echo "<pre>\n";
                    echo str_repeat('-', 160)."\n";
                    printf('SQL: %s'."<br />\n", print_r($sql, true));
                    echo "</pre>\n";
                }

                Db::getInstance()->execute($sql);
            }
        }
    }

    public function init()
    {
        ob_start();
        register_shutdown_function(array('AmazonListOrder', 'jsonDisplayExit'));

        // Identify the order's country
        //
        $marketplace_ids = unserialize(AmazonTools::decode(Configuration::get('AMAZON_MARKETPLACE_ID')));
        $this->marketplace_iso2id = array_flip($marketplace_ids);

        // Regions
        //
        $this->marketplace_region = unserialize(AmazonTools::decode(Configuration::get('AMAZON_REGION')));
        $this->marketplace_lang2region = array_flip($this->marketplace_region);
        $this->marketplace_id_lang = (int)Tools::getValue('amazon_lang');

        if ((int)Tools::getValue('europe')) {
            $masterMarketplace = unserialize(AmazonTools::decode(Configuration::get('AMAZON_MASTER')));

            if (isset($this->marketplace_lang2region[$masterMarketplace]) && $this->marketplace_lang2region[$masterMarketplace]) {
                $this->marketplace_id_lang = $this->marketplace_lang2region[$masterMarketplace];
            } else {
                $this->marketplace_id_lang = (int)Configuration::get('PS_LANG_DEFAULT');
            }

            $this->europe = 1;
        } else {
            $this->europe = 0;
        }

        if (!(int)$this->marketplace_id_lang) {
            AmazonListOrder::$errors[] = $this->l('No selected language, nothing to do...');
            die;
        }

        //  Check Access Tokens
        //
        $tokens = Tools::getValue('amazon_token');

        if (!AmazonTools::checkToken($tokens)) {
            AmazonListOrder::$errors[] = $this->l('Wrong Token');
        }

        // Init
        //
        if ($this->europe) {
            $this->marketplace_setup = AmazonTools::selectEuropeanPlatforms($this->_debug);
        } else {
            $this->marketplace_setup = AmazonTools::selectPlatforms($this->marketplace_id_lang, $this->_debug);
        }
    }
    
    public function displayList()
    {
        $cr = '<br />'; // carriage return
        $date_max = null;

        if ($this->orders_reports_management) {
            $sql = 'SELECT MAX(`date`) as date_max FROM `'._DB_PREFIX_.Amazon::TABLE_MARKETPLACE_ORDER_ADDRESS.'`';
            $result = Db::getInstance()->getValue($sql);

            if (Tools::strlen($result)) {
                $date_max = date('Y-m-d', min(time(), strtotime($result)));
            }
        }

        $current_version = Configuration::get('AMAZON_CURRENT_VERSION', null, 0, 0);

        if (version_compare($current_version, $this->version, '<')) {
            //AmazonListOrder::$errors[] = $this->l('Module version and configuration mismatch, please edit and save your module configuration').$cr;
            //die;
        }

        $currentDate = date('Y-m-d H:i:s');

        AmazonListOrder::$messages[] = $this->l('Starting Order Query in WS API/Web Mode').' - '.$currentDate.$cr;

        $status = (string)Tools::getValue('statuses');

        $date1 = str_replace('-', '/', Tools::getValue('datepickerFrom'));
        $date2 = str_replace('-', '/', Tools::getValue('datepickerTo'));
        $channel = Tools::getValue('channel');

        if (!in_array($channel, array(self::AFN, self::MFN))) {
            $channel = null;
        }

        // Today - 5 minutes de temps de latence afin d'�viter les erreur de synchro dus a l'heure systeme
        //
        if ($date_max) {
            $date1 = date('c', min(strtotime($date1.' 00:00'), strtotime($date_max.' 00:00')));
            $date2 = date('c', min(strtotime($date2.' 23:59:59'), strtotime($date_max.' 23:59:59')));
        } elseif (date('Ymd', strtotime($date2)) >= date('Ymd') || empty($date2)) {
            $date1 = date('c', strtotime($date1));
            $date2 = date('c', strtotime('now - 5 min'));
        } elseif (date('Ymd', strtotime($date1)) >= date('Ymd', strtotime($date2))) {
            $date1 = date('c', strtotime($date1.' 00:00'));
            $date2 = date('c', strtotime($date2.' 23:59:59'));
        } else {
            $date1 = date('c', strtotime($date1.' 00:00'));
            $date2 = date('c', strtotime($date2.' 23:59:59'));
        }

        $tokenOrders = Tools::getValue('token_order');

        if ($status == AmazonOrder::ORDER_IN_CART) {
            $status = AmazonOrder::ORDER_PENDING;
        }

        if (!($this->_amazonApi = new AmazonWebService($this->marketplace_setup['auth'], $this->marketplace_setup['params'], $this->marketplace_setup['platforms'], $this->_debug))) {
            AmazonListOrder::$errors[] = $this->l('Unable to login');
            die;
        }

        if ($this->_debug) {
            echo nl2br(print_r($this->marketplace_setup['auth'], true).print_r($this->marketplace_setup['params'], true).print_r($this->marketplace_setup['platforms'], true));
        }

        // Fix the server's clock drift
        $result = $this->_amazonApi->ServiceStatus(true);

        if (isset($result->GetServiceStatusResult)) {
            if (isset($result->GetServiceStatusResult->Timestamp) && strtotime($result->GetServiceStatusResult->Timestamp) > strtotime('today midnight')) {
                $to_date = min(strtotime((string)$result->GetServiceStatusResult->Timestamp), strtotime($date2));
                $date2 = date('c', $to_date);
            }
        }

        if ($this->amazon_features['demo_mode']) {
            $this->_amazonApi->demo = true;
        }
        $no_pending_order_string = $this->l('No pending order for the selected period');

        // Listing Orders
        //
        $orders = $this->_amazonApi->GetUnshippedOrdersListv4($date1, $date2, $status, $channel);

        if (!$orders || !count($orders)) {
            AmazonListOrder::$warnings[] = sprintf('%s (%d/%s)', $no_pending_order_string, __LINE__, 'List Orders');
            die;
        }

        $orderCheck = new AmazonOrder();

        foreach ($orders as $key => $order) {
            $purchased_date = isset($order->PurchaseDate) ? strtotime($order->PurchaseDate) : null;
            
            if ($purchased_date && $purchased_date < strtotime($date1) || $purchased_date > strtotime($date2)) {
                continue;
            }

            // Langue de la Commande
            //
            if (is_array($this->marketplace_iso2id) && count($this->marketplace_iso2id) && isset($this->marketplace_iso2id[(string)$order->MarketPlaceId])) {
                $this->marketplace_id_lang = $this->marketplace_iso2id[(string)$order->MarketPlaceId];
            } else {
                $this->marketplace_id_lang = $this->context->language->id;
            }

            $pass = true;


            switch ((string)$order->OrderStatus) {
                case AmazonOrder::ORDER_PENDING:
                    $status_string = $this->l('In Cart');

                    if ($status != 'All' && $status != AmazonOrder::ORDER_IN_CART) {
                        $pass = false;
                    }
                    break;

                default:
                    $status_string = (string)$order->OrderStatus;
                    break;
            }

            switch ($status) {
                case AmazonOrder::ORDER_PENDING:
                    if (!in_array((string)$order->OrderStatus, array(AmazonOrder::ORDER_UNSHIPPED, AmazonOrder::ORDER_PARTIALLYSHIPPED))) {
                        $pass = false;
                    }
                    break;
                case AmazonOrder::ORDER_SHIPPED:
                    if (!in_array((string)$order->OrderStatus, array(AmazonOrder::ORDER_SHIPPED))) {
                        $pass = false;
                    }
                    break;
                case AmazonOrder::ORDER_PARTIALLYSHIPPED:
                    if (!in_array($order->OrderStatus, array(AmazonOrder::ORDER_PARTIALLYSHIPPED))) {
                        $pass = false;
                    }
                    break;
                case AmazonOrder::ORDER_UNSHIPPED:
                    if (!in_array($order->OrderStatus, array(AmazonOrder::ORDER_UNSHIPPED))) {
                        $pass = false;
                    }
                    break;
                case AmazonOrder::ORDER_CANCELED:
                    if (!in_array($order->OrderStatus, array(AmazonOrder::ORDER_CANCELED))) {
                        $pass = false;
                    }
                    break;
                case 'All':
                    $pass = true;
                    break;
            }

            if (!$pass) {
                continue;
            }

            $currency = (int)Currency::getIdByIsoCode((string)$order->OrderTotalCurrency);

            $total = (float)$order->OrderTotalAmount;
            $retrieved = $orderCheck->checkByMpId($order->AmazonOrderId);

            $url = '?tab=AdminOrders&id_order='.$retrieved.'&vieworder&token='.$tokenOrders;

            $order_link = $retrieved ? ('<a href="'.$url.'" title="" target="_blank" >'.$order->AmazonOrderId.'('.$retrieved.')</a>') : $order->AmazonOrderId;

            if ($this->marketplace_id_lang) {
                $flag = '<img src="'.$this->images.'geo_flags/'.$this->marketplace_region[$this->marketplace_id_lang].'.jpg" alt="" />';
            } else {
                $flag = '';
            }

            $oID = (string)$order->AmazonOrderId;
            self::$orders[$oID]['id'] = (string)$order->AmazonOrderId;
            self::$orders[$oID]['flag'] = $flag;
            self::$orders[$oID]['date'] = AmazonTools::displayDate(date('Y-m-d H:i:s', strtotime($order->PurchaseDate)), $this->id_lang);
            self::$orders[$oID]['id_lang'] = $this->marketplace_id_lang;
            self::$orders[$oID]['link'] = $order_link;
            self::$orders[$oID]['status'] = $status_string;
            self::$orders[$oID]['imported'] = (bool)$retrieved;
            self::$orders[$oID]['pending'] = (string)$order->OrderStatus == AmazonOrder::ORDER_PENDING;
            self::$orders[$oID]['canceled'] = (string)$order->OrderStatus == AmazonOrder::ORDER_CANCELED;
            self::$orders[$oID]['customer'] = htmlspecialchars((string)$order->Address->Name);
            self::$orders[$oID]['shipping'] = (string)$order->ShipServiceLevel;

            if ($order->IsPrime) {
                self::$orders[$oID]['fulfillment'] = sprintf('%s <b>(Prime)</b>', (string)$order->FulfillmentChannel);
            } else {
                self::$orders[$oID]['fulfillment'] = (string)$order->FulfillmentChannel;
            }

            self::$orders[$oID]['quantity'] = (int)((int)$order->NumberOfItemsUnshipped + (int)$order->NumberOfItemsShipped);
            self::$orders[$oID]['total'] = Tools::displayPrice($total, $currency);
        }
    }

    public function l($string, $specific = false, $id_lang = null)
    {
        return (parent::l($string, basename(__FILE__, '.php'), $id_lang));
    }
}

$amazonOrders = new AmazonListOrder();
$amazonOrders->dispatch(Tools::getValue('action', 'list'));