<?php
/**
 * NOTICE OF LICENSE
 *
 * This source file is subject to a commercial license from Common-Services Co., Ltd.
 * Use, copy, modification or distribution of this source file without written
 * license agreement from the SARL SMC is strictly forbidden.
 * In order to obtain a license, please contact us: contact@common-services.com
 * ...........................................................................
 * INFORMATION SUR LA LICENCE D'UTILISATION
 *
 * L'utilisation de ce fichier source est soumise a une licence commerciale
 * concedee par la societe Common-Services Co., Ltd.
 * Toute utilisation, reproduction, modification ou distribution du present
 * fichier source sans contrat de licence ecrit de la part de la Common-Services Co. Ltd. est
 * expressement interdite.
 * Pour obtenir une licence, veuillez contacter Common-Services Co., Ltd. a l'adresse: contact@common-services.com
 *
 * @author    Olivier B.
 * @copyright Copyright (c) 2011-2018 Common Services Co Ltd - 90/25 Sukhumvit 81 - 10260 Bangkok - Thailand
 * @license   Commercial license
 * @package   Amazon Market Place
 * Support by mail:  support.amazon@common-services.com
 */

require_once(dirname(__FILE__).'/env.php');

require_once(dirname(__FILE__).'/../classes/amazon.tools.class.php');
require_once(dirname(__FILE__).'/../classes/amazon.webservice.class.php');
require_once(dirname(__FILE__).'/../classes/amazon.specificfield.class.php');
require_once(dirname(__FILE__).'/../classes/amazon.valid_values.class.php');

require_once(dirname(__FILE__).'/../amazon.php');

require_once(dirname(__FILE__).'/../validate/Node.php');
require_once(dirname(__FILE__).'/../validate/XmlDataType.php');
require_once(dirname(__FILE__).'/../validate/XmlRestriction.php');
require_once(dirname(__FILE__).'/../validate/AmazonValidator.php');
require_once(dirname(__FILE__).'/../validate/AmazonXSD.php');
require_once(dirname(__FILE__).'/../validate/Tools.php');

class AmazonXSDOperations extends Amazon
{
    private $productType;
    private $universe;
    public $debug;

    public function __construct()
    {
        parent::__construct();

        AmazonContext::restore($this->context);

        if ((int)Tools::getValue('id_lang')) {
            $this->id_lang = (int)Tools::getValue('id_lang');
        }

        if (Amazon::$debug_mode) {
            @ini_set('display_errors', 'on');
            @error_reporting(E_ALL | E_STRICT);
        }
    }

    public function dispatch()
    {
        $callback = Tools::getValue('callback');
        if ($callback == '?') {
            $callback = 'jsonp_'.time();
        }

        ob_start();
        if (Amazon::$debug_mode) {
            echo "<pre>\n";
            printf('%s:%s - ', basename(__FILE__), __LINE__);
            echo "</pre>\n";
        }

        switch (Tools::getValue('action')) {
            case 'load':
                $this->loadSpecificsFields();
                break;
            case 'fetch':
                $this->fetchProductTypes();
                break;
            case 'extrafields':
                $this->extraFields();
                break;
            default:
                die($this->l('No action was choose'));
        }
        $console = ob_get_clean();

        $data = array('fields' => null, 'errors' => false, 'console' => $console);

        echo $callback.'('.Tools::jsonEncode($data).')';
        die;
    }

    public function loadSpecificsFields()
    {
        $callback = Tools::getValue('callback');

        if ($callback == '?') {
            $callback = 'jsonp_'.time();
        }

        $profile_key = Tools::getValue('profile_key');
        $id_lang = Tools::getValue('id_lang');

        $profiles = AmazonConfiguration::get('profiles');

        if (is_array($profiles) && count($profiles)) {
            $profiles = AmazonSpecificField::migrateProfilesFromV3($profiles);

            if (!isset($profiles['name'])) {
                $profiles['name'] = array();
            }

            foreach ($profiles['name'] as $profile_id => $profile_name) {
                if (!Tools::strlen($stored_profile_key = AmazonTools::toKey($profile_name)) && $profile_id != 65535) {
                    continue;
                }

                if ($stored_profile_key != $profile_key) {
                    continue;
                }

                $p_extra = isset($profiles['extra'][$profile_key][$id_lang]) ? $profiles['extra'][$profile_key][$id_lang] : null;

                if (is_array($p_extra) && count($p_extra)) {
                    echo AmazonSpecificField::displayFields($id_lang, $profile_name, $p_extra);
                    die;
                }
            }
        }
        $console = ob_get_clean();

        $data = array('error' => true, 'result' => null, 'console' => $console);

        echo $callback.'('.Tools::jsonEncode($data).')';
        die;
    }

    public function fetchProductTypes()
    {
        $this->errors[] = array();
        $data = array();
        $lang = Tools::getValue('lang');

        if (!($selected = Tools::getValue('selected'))) {
            $this->errors[] = sprintf('%s/%d: %s', basename(__FILE__), __LINE__, $this->l('Missing Parameter: selected'));

            return (false);
        }

        $file = $selected.'.xsd';

        if (!$file) {
            $this->errors[] = sprintf('%s/%d: %s', basename(__FILE__), __LINE__, $this->l('Unable to find the XSD file').': '.$selected);

            return (false);
        }

        //$exclude = array('Home');
        $exclude = array();

        $customElements = array(
            'ProductClothing' => array('ClothingType' => 'Clothing->ClassificationData->ClothingType')
        );

        //get ProductTypes

        $arr = array($selected);

        foreach ($arr as $key => $category) {
            $productTypes = array();

            try {
                $productFactory = new AmazonXSD($category.'.xsd');
                $productInstance = $productFactory->getInstance();

                if ($category == 'CE') {
                    $productType = 'ProductSubtype';
                } else {
                    $productType = 'ProductType';
                }

                $item = new stdClass();
                $item->productType = AmazonXSD::getVariationData($productInstance, $productType);

                if ($item->productType === false) {
                    if ($category == 'Shoes') {
                        $item->productType = AmazonXSD::getVariationData($productInstance, 'ClothingType');
                    }
                }

                if (isset($customElements[$category]) && $item) {
                    foreach ($customElements[$category] as $keyName => $el) {
                        $e = explode('->', $el);
                        $obj = $productInstance->ProductData;

                        foreach ($e as $tag) {
                            $obj = $obj->$tag;
                        }

                        $item->$keyName = $obj;
                    }
                }

                if (!$item) {
                    $item = AmazonXSD::getVariationData($productInstance, 'ProductType');
                }

                if (isset($productInstance->ProductData->Shoes->ClothingType->allowedValues)) {
                    foreach ($productInstance->ProductData->Shoes->ClothingType->allowedValues as $val) {
                        $productTypes[] = (string)$val;
                    }
                } elseif (isset($productInstance->ProductData->Clothing->ClassificationData->ClothingType->allowedValues)) {
                    foreach ($productInstance->ProductData->Clothing->ClassificationData->ClothingType->allowedValues as $val) {
                        $productTypes[] = (string)$val;
                    }
                } elseif (isset($item->productType->allowedValues)) {
                    // Added 2012-01-04 O.B.
                    foreach ($item->productType->allowedValues as $val) {
                        $productTypes[] = (string)$val;
                    }
                } elseif (isset($item->productType) && !empty($item->productType)) {
                    foreach ($item->productType as $key => $val) {
                        if (!preg_match('/^[A-Z]/', (string)$key)) {
                            continue;
                        }

                        $productTypes[] = (string)$key;
                    }
                } elseif (isset(AmazonXSD::$product_type_duplicated_exception[$selected])) {
                    $productTypes[] = $selected;
                } else {
                    // No Types
                    $productTypes = null;
                }
            } catch (Exception $e) {
                $this->errors[] = sprintf('XML Error:'.$e->getMessage());
            }
        }


        if (count($productTypes)) {
            echo '<option value="">'.$this->l('Choose').'</option>';

            foreach ($productTypes as $productType) {
                if (in_array($productType, $exclude)) {
                    continue;
                }

                $translation = null;

                if (Tools::strlen($lang)) {
                    $translation = AmazonSettings::getProductTypeTranslation($lang, $selected, $productType);
                }

                if ($translation) {
                    echo '<option value="'.$productType.'" >'.sprintf('%s (%s)', $translation, $productType).'</option>'."\n";
                } else {
                    echo '<option value="'.$productType.'" >'.$productType.'</option>'."\n";
                }
            }
        } else {
            echo '<option value="">'.$this->l('No Data').'</option>';
        }

        die;
    }

    public function l($string, $specific = false, $id_lang = null)
    {
        return (parent::l($string, basename(__FILE__, '.php'), $id_lang));
    }

    /*
     * Amazon specific fields
     */

    public function extraFields()
    {
        $profile = urldecode(Tools::getValue('profile'));
        $universe = Tools::getValue('universe');
        $productType = Tools::getValue('selected');
        $callback = Tools::getValue('callback');
        $specificFields = array();
        $specificFields[$profile]['universe'] = $universe;
        $specificFields[$profile]['product_type'] = $productType;
        $specificFields[$profile]['version'] = '4.0';

        if ($callback == '?') {
            $callback = 'jsonp_'.time();
        }

        if (!Tools::strlen($universe) || !Tools::strlen($productType)) {
            die;
        }

        $productFactory = new AmazonXSD($universe.'.xsd');
        $p = $productFactory->getInstance();

        if ($universe == 'SWVG') {
            $universe = 'SoftwareVideoGames';
        }

        $this->universe = $universe;
        $this->productType = $productType;

        $required = AmazonXSD::getMandatoryFields($p->ProductData);
        $errors = ob_get_contents();

        if (Amazon::$debug_mode) {
            echo "<pre>Parameters\n";
            printf('%s(%d)'."\n", basename(__FILE__), __LINE__);
            printf("Universe: %s\n", $universe);
            printf("Product Type: %s\n", $productType);
            echo "</pre>\n";

            echo "<pre>Required Fields\n";
            var_dump($required);
            echo "</pre>\n";
        }

        /*
         * Parse Variations/Variation Themes
         */
        $specificFields[$profile]['variation'] = array();

        if ($specificFields[$profile]['variation'] = AmazonXSDTools::parseVariationData($p, $universe, $productType)) {
            foreach ($specificFields[$profile]['variation'] as $variationThemeItem) {
                if (is_array($variationThemeItem) && count($variationThemeItem)) {
                    foreach ($variationThemeItem['fields'] as $variationField) {
                        $specificFields[$profile]['field'][$variationField] = null;

                        if (isset($variationThemeItem['attributes']) && isset($variationThemeItem['attributes'][$variationField])) {
                            $specificFields[$profile]['attributes'][$variationField] = array();
                            $specificFields[$profile]['attributes'][$variationField]['unitOfMeasure'] = true;
                            $specificFields[$profile]['values'][$variationField] = $variationThemeItem['attributes'][$variationField];
                        }
                    }
                }
            }
        }

        if (Amazon::$debug_mode) {
            echo "<pre>Variation\n";
            printf('%s(%d)'."\n", basename(__FILE__), __LINE__);
            var_dump($specificFields[$profile]['variation']);
            echo "</pre>\n";
        }


        if ($universe == 'Beauty') {
            // Optionnal Settings
            //
            $this->optionnalFields($specificFields[$profile], $p->ProductData->Beauty);

            $this->otherFields($specificFields[$profile], $p->ProductData->Beauty);
        } elseif ($universe == 'Luggage') {
            $specificFields[$profile]['field']['Department'] = null;
            $specificFields[$profile]['required']['Department'] = true;

            $specificFields[$profile]['field']['Weight'] = null;
            $specificFields[$profile]['required']['Weight'] = true;

            $specificFields[$profile]['field']['VolumeCapacityName'] = null;
            $specificFields[$profile]['required']['VolumeCapacityName'] = true;

            // Optionnal Settings
            //
            $this->optionnalFields($specificFields[$profile], $p->ProductData->Luggage);

            $this->otherFields($specificFields[$profile], $p->ProductData->Luggage);
        } elseif ($universe == 'ClothingAccessories') {
            // ClothingAccessories
            $specificFields[$profile]['field']['Department'] = null;
            $specificFields[$profile]['required']['Department'] = true;

            // Optionnal Settings
            //
            $this->optionnalFields($specificFields[$profile], $p->ProductData->ClothingAccessories->ClassificationData);

            $this->otherFields($specificFields[$profile], $p->ProductData->ClothingAccessories->ClassificationData);
        } elseif (isset($p->ProductData->Shoes->ClassificationData->Department)) {
            // Shoes
            $specificFields[$profile]['field']['Department'] = null;
            $specificFields[$profile]['required']['Department'] = true;

            // Optionnal Settings
            //
            $this->optionnalFields($specificFields[$profile], $p->ProductData->Shoes->ClassificationData);

            $this->otherFields($specificFields[$profile], $p->ProductData->Shoes->ClassificationData);
        } elseif (isset($p->ProductData->Clothing->ClassificationData)) {
            // Clothes
            $specificFields[$profile]['field']['Department'] = null;
            $specificFields[$profile]['required']['Department'] = true;
            $specificFields[$profile]['field']['CareInstructions'] = null;

            // Optionnal Settings
            //
            $this->optionnalFields($specificFields[$profile], $p->ProductData->Clothing->ClassificationData);

            $this->otherFields($specificFields[$profile], $p->ProductData->Clothing->ClassificationData);
        } elseif (isset($p->ProductData->Jewelry)) {
            // Jewelry
            $specificFields[$profile]['field']['DepartmentName'] = null;

            // Optionnal Settings
            //
            $this->optionnalFields($specificFields[$profile], $p->ProductData->$universe->ProductType->$productType);

            // Optionnal Settings
            //
            $this->otherFields($specificFields[$profile], $p->ProductData->$universe->ProductType->$productType);

            if (isset($p->ProductData->$universe->ProductType->$productType->Stone)) {
                // Extra-Optionnal Settings
                //
                $this->otherFields($specificFields[$profile], $p->ProductData->$universe->ProductType->$productType->Stone);
            }
        } elseif (isset($p->ProductData->FoodAndBeverages)) {
            // Optionnal Settings
            //
            $this->optionnalFields($specificFields[$profile], $p->ProductData->$universe->ProductType->$productType);

            // Optionnal Settings
            //
            $this->otherFields($specificFields[$profile], $p->ProductData->$universe->ProductType->$productType->NutritionalFactsGroup);

            // Optionnal Settings
            //
            $this->otherFields($specificFields[$profile], $p->ProductData->$universe->ProductType->$productType);
        } elseif (isset($p->ProductData->Music)) {
            // Optionnal Settings
            //
            $this->optionnalFields($specificFields[$profile], $p->ProductData->$universe->ProductType->$productType);

            if (isset($specificFields[$profile], $p->ProductData->$universe->ProductType->$productType->VinylRecordDetails)) {
                // Optionnal Settings
                //
                $this->otherFields($specificFields[$profile], $p->ProductData->$universe->ProductType->$productType->VinylRecordDetails);
            }

            // Optionnal Settings
            //
            $this->otherFields($specificFields[$profile], $p->ProductData->$universe->ProductType->$productType);
        } elseif (isset($p->ProductData->$universe->ProductType->$productType) && $p->ProductData->$universe->ProductType->$productType instanceof stdClass) {
            // most products

            $f = 0;

            foreach ($p->ProductData->$universe->ProductType->$productType as $name => $tag) {
                if (isset($specificFields[$profile]['field'][$name])) {
                    continue;
                }
                if (AmazonXSD::isMandatoryField($tag) && $tag instanceof stdClass && isset($tag->type)) {
                    if (in_array($tag->type, array('text', 'number', 'boolean', 'dateTime'))) {
                        $specificFields[$profile]['field'][$name] = null;
                        $specificFields[$profile]['required'][$name] = true;
                        $specificFields[$profile]['type'][$name] = $tag->type;

                        if (isset($tag->attr->unitOfMeasure) && isset($tag->attr->unitOfMeasure->allowedValues) && count($tag->attr->unitOfMeasure->allowedValues)) {
                            $specificFields[$profile]['attributes'][$name] = array();
                            $specificFields[$profile]['attributes'][$name]['unitOfMeasure'] = true;
                            $specificFields[$profile]['values'][$name] = AmazonTools::encode(serialize($tag->attr->unitOfMeasure->allowedValues));
                        }
                        $f++;
                    }
                }

                // choice
                if (isset($tag->choice) && $tag->choice) {
                    $specificFields[$profile]['choices_required'] = $tag->mandatory;
                    $specificFields[$profile]['choice_allowed_values'][$name] =  AmazonTools::encode(serialize($tag->allowedValues));
                }
            }

            // Optionnal Settings
            //
            $this->optionnalFields($specificFields[$profile], $p->ProductData->$universe->ProductType->$productType);

            $this->otherFields($specificFields[$profile], $p->ProductData->$universe->ProductType->$productType);

            $this->otherFields($specificFields[$profile], $p->ProductData->$universe);

            // Some Cases (eg: Toys - Minimum Age)
            //
            foreach ($required as $requirements) {
                if (is_object($requirements)) {
                    foreach ($requirements as $key => $tag) {
                        if ($key == 'ProductType') {
                            continue;
                        }

                        if ($tag instanceof stdClass) {
                            if (AmazonXSD::isParentWithChildren($key)) {
                                $children = get_object_vars($requirements->$key);
                                foreach ($children as $keyName => $tagValue) {
                                    if ($tagValue instanceof stdClass) {
                                        $specificFields[$profile]['field'][$keyName] = null;
                                        $specificFields[$profile]['type'][$keyName] = isset($tag->type) ? $tag->type : null;
                                        $specificFields[$profile]['required'][$keyName] = true;

                                        if (isset($tagValue->attr->unitOfMeasure) && isset($tagValue->attr->unitOfMeasure->allowedValues) && count($tagValue->attr->unitOfMeasure->allowedValues)) {
                                            $specificFields[$profile]['attributes'][$keyName] = array();
                                            $specificFields[$profile]['attributes'][$keyName]['unitOfMeasure'] = 0;
                                            $specificFields[$profile]['values'][$keyName] = AmazonTools::encode(serialize($tagValue->attr->unitOfMeasure->allowedValues));
                                        }
                                    }
                                }
                            } else {
                                $specificFields[$profile]['field'][$key] = null;
                                $specificFields[$profile]['type'][$key] = isset($tag->type) ? $tag->type : null;
                                $specificFields[$profile]['required'][$key] = true;

                                if (isset($tag->attr->unitOfMeasure) && isset($tag->attr->unitOfMeasure->allowedValues) && count($tag->attr->unitOfMeasure->allowedValues)) {
                                    $specificFields[$profile]['attributes'][$key] = array();
                                    $specificFields[$profile]['attributes'][$key]['unitOfMeasure'] = 0;
                                    $specificFields[$profile]['values'][$key] = AmazonTools::encode(serialize($tag->attr->unitOfMeasure->allowedValues));
                                }
                            }
                        }
                    }
                }
            }
        } elseif (isset($p->ProductData->$universe->ProductType->allowedValues) && in_array($productType, $p->ProductData->$universe->ProductType->allowedValues)) {
            // Other Products

            foreach ($p->ProductData->$universe as $name => $tag) {
                if (isset($specificFields[$profile]['field'][$name])) {
                    continue;
                }
                if ($name == 'ProductType') {
                    continue;
                }

                if ($tag instanceof stdClass && isset($tag->type)) {
                    if (in_array($tag->type, array('text', 'number', 'boolean', 'dateTime'))) {
                        $specificFields[$profile]['field'][$name] = null;
                        $specificFields[$profile]['type'][$name] = $tag->type;

                        if (isset($tag->attr->unitOfMeasure) && isset($tag->attr->unitOfMeasure->allowedValues) && count($tag->attr->unitOfMeasure->allowedValues)) {
                            $specificFields[$profile]['attributes'][$name] = array();
                            $specificFields[$profile]['attributes'][$name]['unitOfMeasure'] = true;
                            $specificFields[$profile]['values'][$name] = AmazonTools::encode(serialize($tag->attr->unitOfMeasure->allowedValues));
                        }
                    }
                }
            }
        }

        echo AmazonSpecificField::displayFields($this->id_lang, $profile, $specificFields[$profile]);


        if (Amazon::$debug_mode) {
            echo "<pre>\n";
            printf('%s(%d)'."\n", basename(__FILE__), __LINE__);
            var_dump($specificFields);
            echo "</pre>\n";
        }

        $fields = ob_get_clean();
        $data = array('fields' => $fields, 'errors' => $errors);

        echo $callback.'('.Tools::jsonEncode(self::fixEncoding($data)).')';
        die;
    }

    public static function fixEncoding($array_to_fix)
    {
        if (is_array($array_to_fix) && count($array_to_fix)) {
            foreach ($array_to_fix as $key => $item) {
                if (!mb_check_encoding($item, 'UTF-8')) {
                    $array_to_fix[$key] = mb_convert_encoding($item, "UTF-8");
                }
            }
        }

        return ($array_to_fix);
    }

    /*
     * Fetch categories
     */

    public function optionnalFields(&$specificFields, $productDefinition = null)
    {
        if (count($productDefinition)) {
            $universe = $this->universe == 'ClothingAccessories' ? 'ProductClothing' : $this->universe;

            $default_fields = AmazonValidValues::getAttributesForUniverse($universe);

            foreach ($default_fields as $key => $default_field) {
                if (!Tools::strlen($default_field)) {
                    continue;
                }

                if (!property_exists($productDefinition, $default_field)) {
                    unset($default_fields[$key]);
                    continue;
                } else {
                    $specificFields['has_valid_values'][$default_field] = true;
                }
            }
            $default_fields = array_values($default_fields);
        } else {
            $default_fields = AmazonValidValues::getAttributesForProductType($this->universe, $this->productType);
        }

        if (is_array($default_fields) && count($default_fields)) {
            foreach ($default_fields as $field) {
                if (!isset($productDefinition->$field)) {
                    continue;
                }

                $specificFields['field'][$field] = null;
                $specificFields['type'][$field] = isset($productDefinition->{$field}->type) ? $productDefinition->{$field}->type : null;
                $specificFields['attributes'][$field] = array();

                if (isset($productDefinition->$field->attr->unitOfMeasure) && isset($productDefinition->$field->attr->unitOfMeasure->allowedValues) && count($productDefinition->$field->attr->unitOfMeasure->allowedValues)) {
                    $specificFields['attributes'][$field]['unitOfMeasure'] = true;
                    /*
                     * Serialize possible values for the form
                     * MM, CM, etc...
                     */
                    $specificFields['values'][$field] = AmazonTools::encode(serialize($productDefinition->$field->attr->unitOfMeasure->allowedValues));
                }
            }
        }

        if (isset(AmazonXSD::$recommendedPerUniverseFields[$this->universe])) {
            foreach (AmazonXSD::$recommendedPerUniverseFields[$this->universe] as $field) {
                if (isset($specificFields['field'][$field])) {
                    continue;
                }
                if (!isset($productDefinition->$field)) {
                    continue;
                }

                $specificFields['field'][$field] = null;
                $specificFields['type'][$field] = isset($productDefinition->{$field}->type) ? $productDefinition->{$field}->type : null;
                $specificFields['attributes'][$field] = array();

                if (isset($productDefinition->$field->attr->unitOfMeasure) && isset($productDefinition->$field->attr->unitOfMeasure->allowedValues) && count($productDefinition->$field->attr->unitOfMeasure->allowedValues)) {
                    $specificFields['attributes'][$field]['unitOfMeasure'] = true;
                    /*
                     * Serialize possible values for the form
                     * MM, CM, etc...
                     */
                    $specificFields['values'][$field] = AmazonTools::encode(serialize($productDefinition->$field->attr->unitOfMeasure->allowedValues));
                }
            }
        }

        if (isset(AmazonXSD::$recommendedPerTypeFields[$this->productType])) {
            foreach (AmazonXSD::$recommendedPerTypeFields[$this->productType] as $field) {
                if (isset($specificFields['field'][$field])) {
                    continue;
                }
                if (!isset($productDefinition->$field)) {
                    continue;
                }

                $specificFields['field'][$field] = null;
                $specificFields['type'][$field] = isset($productDefinition->{$field}->type) ? $productDefinition->{$field}->type : null;
                $specificFields['attributes'][$field] = array();

                if (isset($productDefinition->$field->attr->unitOfMeasure) && isset($productDefinition->$field->attr->unitOfMeasure->allowedValues) && count($productDefinition->$field->attr->unitOfMeasure->allowedValues)) {
                    $specificFields['attributes'][$field]['unitOfMeasure'] = true;
                    /*
                     * Serialize possible values for the form
                     * MM, CM, etc...
                     */
                    $specificFields['values'][$field] = AmazonTools::encode(serialize($productDefinition->$field->attr->unitOfMeasure->allowedValues));
                }
            }
        }

        return (true);
    }

    public function otherFields(&$specificFields, $productDefinition = null)
    {
        foreach ($productDefinition as $name => $tag) {
            if (isset($specificFields['field'][$name])) {
                continue;
            }

            if ($tag instanceof stdClass && isset($tag->type)) {
                if (in_array($tag->type, array('text', 'number', 'boolean', 'dateTime'))) {
                    $specificFields['field'][$name] = null;
                    $specificFields['type'][$name] = $tag->type;

                    if (isset($tag->attr->unitOfMeasure) && isset($tag->attr->unitOfMeasure->allowedValues) && count($tag->attr->unitOfMeasure->allowedValues)) {
                        $specificFields['attributes'][$name] = array();
                        $specificFields['attributes'][$name]['unitOfMeasure'] = true;
                        $specificFields['values'][$name] = AmazonTools::encode(serialize($tag->attr->unitOfMeasure->allowedValues));
                    }
                }
            }
        }
    }
}

$amazonXSDOperations = new AmazonXSDOperations();
$amazonXSDOperations->dispatch();
