<?php
/**
 * Marketing Popup Prestashop module
 * 
 * @author    Prestaddons <contact@prestaddons.fr>
 * @copyright 2014 Prestaddons
 * @license
 * @link      http://www.prestaddons.fr
 */

if (!defined('_PS_VERSION_'))
	exit;

class MarketingPopup extends Module
{
	/** @var string html Output */
	protected $html = '';

	/** @var array post_errors Errors on forms */
	protected $post_errors = array();

	/** @var bool $is_version15 Prestashop is under 1.5 version */
	private $is_version15;

	/** @var bool $is_version16 Prestashop is under 1.6 version */
	private $is_version16;

	/**
	 * Constructeur de la classe Marketing Popup
	 */
	public function __construct()
	{
		$this->name = 'marketingpopup';
		$this->short_name = 'mp';
		$this->tab = 'advertising_marketing';
		$this->version = '1.7.2';
		$this->need_instance = 0;
		$this->ps_versions_compliancy = array('min' => '1.4', 'max' => '1.7');
		$this->bootstrap = true;
		$this->module_key = '99bd3ab03d089694e2497471b5d5e1d9';

		parent::__construct();

		$this->displayName = $this->l('Marketing Popup');
		$this->description = $this->l('Boost your marketing operations thanks to a customizable frontend lightbox');
		$this->confirmUninstall = $this->l('Are you sure you want to uninstall the Marketing Popup module?');
		$this->author = $this->l('Prestaddons');
		$this->contact = 'contact@prestaddons.fr';
		$this->is_version15 = $this->checkPSVersion();
		$this->is_version16 = $this->checkPSVersion('1.6.0.0');

		// Rétrocompatibilité 1.4/1.5
		require(_PS_MODULE_DIR_.$this->name.'/backward_compatibility/backward.php');
	}

	/**
	 * Méthode install()
	 * 
	 * Gère l'installation du module
	 * 
	 * @return bool True si l'installation a fonctionné, false dans le cas contraire
	 */
	public function install()
	{
		if ($this->is_version15 && Shop::isFeatureActive())
			Shop::setContext(Shop::CONTEXT_ALL);

		$params = $this->installFixtures();

		if (!parent::install() || !$this->registerHook('header') || !$this->registerHook('top')
				|| !Configuration::updateValue($this->short_name.'_popup_content', rawurlencode(serialize($params['popup_content'])))
				|| !Configuration::updateValue($this->short_name.'_active', 1)
				|| !Configuration::updateValue($this->short_name.'_auto_size', 0)
				|| !Configuration::updateValue($this->short_name.'_popup_width', 400)
				|| !Configuration::updateValue($this->short_name.'_popup_height', 400)
				|| !Configuration::updateValue($this->short_name.'_popup_padding', 15)
				|| !Configuration::updateValue($this->short_name.'_popup_margin', 20)
				|| !Configuration::updateValue($this->short_name.'_disp_close', 1)
				|| !Configuration::updateValue($this->short_name.'_background_color', '#ffffff')
				|| !Configuration::updateValue($this->short_name.'_date_from', date('Y-m-d H:i:s'))
				|| !Configuration::updateValue($this->short_name.'_date_to', date('Y-m-d H:i:s'))
				|| !Configuration::updateValue($this->short_name.'_popup_delay', 0)
				|| !Configuration::updateValue($this->short_name.'_cookie_expiration', 0)
				|| !Configuration::updateValue($this->short_name.'_unit_expiration', 0)
				|| !Configuration::updateValue($this->short_name.'_file_list', serialize(array()))
				|| !Configuration::updateValue($this->short_name.'_mp_social', 0)
				|| !Configuration::updateValue($this->short_name.'_mp_facebook', '')
				|| !Configuration::updateValue($this->short_name.'_mp_twitter', '')
				|| !Configuration::updateValue($this->short_name.'_mp_rss', '')
				|| !Configuration::updateValue($this->short_name.'_mp_youtube', '')
				|| !Configuration::updateValue($this->short_name.'_mp_google', '')
				|| !Configuration::updateValue($this->short_name.'_mp_pinterest', '')
				|| !Configuration::updateValue($this->short_name.'_mp_newsletter', 0))
			return false;

		if ($this->is_version15 && !$this->registerHook('displayMarketingPopupForm'))
			return false;

		$this->generateCSS();

		return true;
	}

	/**
	 * Méthode installFixtures()
	 * 
	 * Initialise tous les paramètres nécessaires à l'installation du module
	 * 
	 * @return array $params Tableau contenant les paramètres nécessaires à l'installation
	 */
	private function installFixtures()
	{
		if ($this->is_version15)
		{
			$id_lang_en = Language::getIdByIso('en');
			$id_lang_fr = Language::getIdByIso('fr');
		}
		else
		{
			$id_lang_en = 1;
			$id_lang_fr = 2;
		}

		$params = array(
			'popup_content' => array(
				$id_lang_en => '<h1>Insert your custom content here !</h1>',
				$id_lang_fr => '<h1>Insérez votre contenu personnalisé ici !</h1>'
			)
		);

		return $params;
	}

	/**
	 * Méthode uninstall()
	 * 
	 * Gère la désinstallation du module
	 * 
	 * @return bool True si la désinstallation a fonctionné, false dans le cas contraire
	 */
	public function uninstall()
	{
		if (!parent::uninstall()
				|| !Configuration::deleteByName($this->short_name.'_popup_content')
				|| !Configuration::deleteByName($this->short_name.'_active')
				|| !Configuration::deleteByName($this->short_name.'_auto_size')
				|| !Configuration::deleteByName($this->short_name.'_popup_width')
				|| !Configuration::deleteByName($this->short_name.'_popup_height')
				|| !Configuration::deleteByName($this->short_name.'_popup_padding')
				|| !Configuration::deleteByName($this->short_name.'_popup_margin')
				|| !Configuration::deleteByName($this->short_name.'_disp_close')
				|| !Configuration::deleteByName($this->short_name.'_background_color')
				|| !Configuration::deleteByName($this->short_name.'_date_from')
				|| !Configuration::deleteByName($this->short_name.'_date_to')
				|| !Configuration::deleteByName($this->short_name.'_popup_delay')
				|| !Configuration::deleteByName($this->short_name.'_cookie_expiration')
				|| !Configuration::deleteByName($this->short_name.'_unit_expiration')
				|| !Configuration::deleteByName($this->short_name.'_file_list')
				|| !Configuration::deleteByName($this->short_name.'_mp_social')
				|| !Configuration::deleteByName($this->short_name.'_mp_facebook')
				|| !Configuration::deleteByName($this->short_name.'_mp_twitter')
				|| !Configuration::deleteByName($this->short_name.'_mp_rss')
				|| !Configuration::deleteByName($this->short_name.'_mp_youtube')
				|| !Configuration::deleteByName($this->short_name.'_mp_google')
				|| !Configuration::deleteByName($this->short_name.'_mp_pinterest')
				|| !Configuration::deleteByName($this->short_name.'_mp_newsletter'))
			return false;

		return true;
	}

	/**
	 * Méthode postValidation()
	 * 
	 * Contrôle les variables saisies dans le backoffice et définit les éventuelles erreurs à afficher
	 * 
	 * @return string HTML du résultat de la vérification (message d'erreur éventuel)
	 */
	private function postValidation()
	{
			if (Tools::isSubmit('submit'.$this->name))
			{
				if (!Tools::getValue('auto_size'))
				{
					if (!Validate::isUnsignedInt(Tools::getValue('popup_width')))
						$this->post_errors[] = $this->l('The popup width must be a positive integer');
					if (!Validate::isUnsignedInt(Tools::getValue('popup_height')))
						$this->post_errors[] = $this->l('The popup height must be a positive integer');
				}
				if (!Validate::isUnsignedInt(Tools::getValue('popup_padding')))
					$this->post_errors[] = $this->l('The popup padding must be a positive integer');
				if (!Validate::isUnsignedInt(Tools::getValue('popup_margin')))
					$this->post_errors[] = $this->l('The popup margin must be a positive integer');
				if (Tools::getValue('background_color') == '')
					$this->post_errors[] = $this->l('Background color must not be empty');
				if (!Validate::isUnsignedInt(Tools::getValue('popup_delay')))
					$this->post_errors[] = $this->l('The popup delay must be a positive integer');
				if (!Validate::isUnsignedInt(Tools::getValue('cookie_expiration')))
					$this->post_errors[] = $this->l('The Cookie expiration time must be a positive integer');
			}
	}

	/**
	 * Méthode postProcess()
	 * 
	 * Traitement des informations saisies dans le backoffice
	 * Traitements divers, mise à jour la base de données, définition des messages d'erreur ou de confirmation...
	 * 
	 * @return string HTML du résultat du traitement (message d'erreur ou de confirmation)
	 */
	private function postProcess()
	{
		if (Tools::isSubmit('submit'.$this->name))
		{
			$languages = Language::getLanguages(false);
			$popup_content = array();
			foreach ($languages as $language)
				$popup_content[$language['id_lang']] = Tools::getValue('popup_content_'.$language['id_lang']);

			Configuration::updateValue($this->short_name.'_popup_content', rawurlencode(serialize($popup_content)));
			Configuration::updateValue($this->short_name.'_active', Tools::getValue('mp_active'));
			Configuration::updateValue($this->short_name.'_auto_size', Tools::getValue('auto_size'));
			Configuration::updateValue($this->short_name.'_popup_width', Tools::getValue('popup_width'));
			Configuration::updateValue($this->short_name.'_popup_height', Tools::getValue('popup_height'));
			Configuration::updateValue($this->short_name.'_popup_padding', Tools::getValue('popup_padding'));
			Configuration::updateValue($this->short_name.'_popup_margin', Tools::getValue('popup_margin'));
			Configuration::updateValue($this->short_name.'_disp_close', Tools::getValue('disp_close'));
			Configuration::updateValue($this->short_name.'_background_color', Tools::getValue('background_color'));
			Configuration::updateValue($this->short_name.'_date_from', Tools::getValue('date_from'));
			Configuration::updateValue($this->short_name.'_date_to', Tools::getValue('date_to'));
			Configuration::updateValue($this->short_name.'_popup_delay', Tools::getValue('popup_delay'));
			Configuration::updateValue($this->short_name.'_cookie_expiration', Tools::getValue('cookie_expiration'));
			Configuration::updateValue($this->short_name.'_unit_expiration', Tools::getValue('unit_expiration'));

			$this->generateCSS();

			$this->html .= $this->displayConfirmation($this->l('Settings have been updated'));
		}
		elseif (Tools::isSubmit('submitFeatures'))
		{
			Configuration::updateValue($this->short_name.'_mp_social', Tools::getValue('mp_social'));
			Configuration::updateValue($this->short_name.'_mp_facebook', Tools::getValue('mp_facebook'));
			Configuration::updateValue($this->short_name.'_mp_twitter', Tools::getValue('mp_twitter'));
			Configuration::updateValue($this->short_name.'_mp_rss', Tools::getValue('mp_rss'));
			Configuration::updateValue($this->short_name.'_mp_youtube', Tools::getValue('mp_youtube'));
			Configuration::updateValue($this->short_name.'_mp_google', Tools::getValue('mp_google'));
			Configuration::updateValue($this->short_name.'_mp_pinterest', Tools::getValue('mp_pinterest'));
			Configuration::updateValue($this->short_name.'_mp_newsletter', Tools::getValue('mp_newsletter'));

			$this->generateCSS();

			$this->html .= $this->displayConfirmation($this->l('Additional features have been updated'));
		}
		elseif (Tools::isSubmit('submitExceptions'))
		{
			$exceptions = Tools::getValue('exceptions');
			if ($exceptions != '')
			{
				$exceptions = (isset($exceptions[0])) ? $exceptions[0] : array();
				$exceptions = explode(',', str_replace(' ', '', $exceptions));
			}
			else
				$exceptions = array();

			Configuration::updateValue($this->short_name.'_file_list', serialize($exceptions));

			$this->generateCSS();

			$this->html .= $this->displayConfirmation($this->l('Exception pages have been updated'));
		}
	}

	/**
	 * Méthode getContent()
	 * 
	 * Gère l'administration du module dans le backoffice
	 * Dispatch vers les différentes méthodes en fonctions des cas (affichage des formulaires, des erreurs, des confirmations, ...)
	 * 
	 * @return string HTML de la partie backoffice du module
	 */
	public function getContent()
	{
		$this->postValidation();

		if (!count($this->post_errors))
			$this->postProcess();
		else
		{
			foreach ($this->post_errors as $err)
				$this->html .= $this->displayError($err);
		}

		if (Tools::isSubmit('support'.$this->name))
			$this->html .= $this->renderSupportForm();
		else
			$this->html .= $this->renderForm();

		return $this->html;
	}

	/**
	 * Méthode renderForm()
	 * 
	 * Affiche le formulaire principale du module dans le backoffice
	 * 
	 * @return string HTML du backoffice du module
	 */
	public function renderForm()
	{
		$this->context->controller->addCSS($this->_path.'css/mp-admin.css');
		$this->context->controller->addJS($this->_path.'js/mp-admin.js');

		if ($this->is_version15)
		{
			// Get default Language
			$default_lang = (int)Configuration::get('PS_LANG_DEFAULT');

			// Init Fields form array
			$fields_form = array();
			$fields_form[0]['form'] = array(
				'legend' => array(
					'title' => $this->l('Settings'),
					(!$this->is_version16) ? 'image' : 'icon' => (!$this->is_version16) ? _MODULE_DIR_.$this->name.'/img/settings_16x16.png' : 'icon-cogs'
				),
				'input' => array(
					array(
						'type' => 'textarea',
						'label' => $this->l('Popup content'),
						'desc' => $this->l('Content displayed in the Marketing Popup'),
						'name' => 'popup_content',
						'autoload_rte' => true,
						'lang' => true,
					),
					array(
						'type' => ($this->is_version16) ? 'switch' : 'radio',
						'label' => $this->l('Enable'),
						'desc' => $this->l('Enable the Marketing Popup'),
						'name' => 'mp_active',
						'class' => 't', //only 1.5
						'is_bool' => true,
						'values' => array(
							array(
								'id' => 'active_on',
								'value' => 1,
								'label' => $this->l('Yes')
							),
							array(
								'id' => 'active_off',
								'value' => 0,
								'label' => $this->l('No')
							)
						),
					),
					array(
						'type' => ($this->is_version16) ? 'text' : 'date',
						'label' => $this->l('From'),
						'desc' => $this->l('Set the beginning date'),
						'name' => 'date_from',
						'suffix' => '<i class="icon-calendar"></i>', //only 1.6
						'required' => false,
						'size' => 20,
						'class' => 'fixed-width-lg datepicker' //only 1.6
					),
					array(
						'type' => ($this->is_version16) ? 'text' : 'date',
						'label' => $this->l('To'),
						'desc' => $this->l('Set the ending date'),
						'name' => 'date_to',
						'suffix' => '<i class="icon-calendar"></i>', //only 1.6
						'required' => false,
						'size' => 20,
						'class' => 'fixed-width-lg datepicker' //only 1.6
					),
					array(
						'type' => ($this->is_version16) ? 'switch' : 'radio',
						'label' => $this->l('Auto size'),
						'desc' => $this->l('Size sets according to the content'),
						'name' => 'auto_size',
						'class' => 't', //only 1.5
						'is_bool' => true,
						'values' => array(
							array(
								'id' => 'auto_size_on',
								'value' => 1,
								'label' => $this->l('Yes')
							),
							array(
								'id' => 'auto_size_off',
								'value' => 0,
								'label' => $this->l('No')
							)
						),
					),
					array(
						'type' => 'text',
						'label' => $this->l('Width'),
						'desc' => $this->l('Set the popup width'),
						'name' => 'popup_width',
						'suffix' => 'px',
						'required' => true,
						'size' => 20,
						'class' => 'fixed-width-lg' //only 1.6
					),
					array(
						'type' => 'text',
						'label' => $this->l('Height'),
						'desc' => $this->l('Set the popup height'),
						'name' => 'popup_height',
						'suffix' => 'px',
						'required' => true,
						'size' => 20,
						'class' => 'fixed-width-lg' //only 1.6
					),
					array(
						'type' => 'text',
						'label' => $this->l('Padding'),
						'desc' => $this->l('Set the popup padding'),
						'name' => 'popup_padding',
						'suffix' => 'px',
						'required' => true,
						'size' => 20,
						'class' => 'fixed-width-lg' //only 1.6
					),
					array(
						'type' => 'text',
						'label' => $this->l('Margin'),
						'desc' => $this->l('Set the popup margin'),
						'name' => 'popup_margin',
						'suffix' => 'px',
						'required' => true,
						'size' => 20,
						'class' => 'fixed-width-lg' //only 1.6
					),
					array(
						'type' => ($this->is_version16) ? 'switch' : 'radio',
						'label' => $this->l('Close button'),
						'desc' => $this->l('Display the close button'),
						'name' => 'disp_close',
						'class' => 't', //only 1.5
						'is_bool' => true,
						'values' => array(
							array(
								'id' => 'disp_close_on',
								'value' => 1,
								'label' => $this->l('Yes')
							),
							array(
								'id' => 'disp_close_off',
								'value' => 0,
								'label' => $this->l('No')
							)
						),
					),
					array(
						'type' => 'color',
						'label' => $this->l('Background color'),
						'desc' => $this->l('Set the popup background color'),
						'name' => 'background_color',
						'required' => true,
						'size' => 20,
						'class' => 'fixed-width-lg color mColorPickerInput' //only 1.6
					),
					array(
						'type' => 'text',
						'label' => $this->l('Delay'),
						'desc' => $this->l('Delay before popup appears'),
						'name' => 'popup_delay',
						'suffix' => 'ms',
						'required' => true,
						'size' => 20,
						'class' => 'fixed-width-lg' //only 1.6
					)
				),
				'submit' => array(
					'title' => $this->l('Save'),
					'class' => ($this->is_version16) ? 'btn btn-default pull-right' : 'button',
					'name' => 'submit'.$this->name
				)
			);

			$helper = new HelperForm();

			// Module, token and currentIndex
			$helper->module = $this;
			$helper->name_controller = $this->name;
			$helper->token = Tools::getAdminTokenLite('AdminModules');
			$helper->currentIndex = AdminController::$currentIndex.'&configure='.$this->name;

			// Language
			$helper->default_form_language = $default_lang;
			$helper->allow_employee_form_lang = $default_lang;

			// Title and toolbar
			$helper->title = $this->displayName;
			$helper->show_toolbar = true;  // false -> remove toolbar. Only 1.5
			$helper->toolbar_scroll = true;   // true -> Toolbar is always visible on the top of the screen. Only 1.5
			if (!$this->checkPSVersion('1.5.5.0'))
				$helper->submit_action = 'submit'.$this->name;

			if (!$this->is_version16)
			{
				$helper->toolbar_btn['save'] = array(
					'desc' => $this->l('Save'),
					'href' => AdminController::$currentIndex.'&configure='.$this->name.'&save'.$this->name.'&token='.Tools::getAdminTokenLite('AdminModules')
				);
				$helper->toolbar_btn['back'] = array(
					'desc' => $this->l('Back to list'),
					'href' => AdminController::$currentIndex.'&token='.Tools::getAdminTokenLite('AdminModules')
				);
			}

			$helper->toolbar_btn['help-new'] = array(
				'desc' => $this->l('Support'),
				'href' => AdminController::$currentIndex.'&configure='.$this->name.'&support'.$this->name.'&token='.Tools::getAdminTokenLite('AdminModules')
			);

			$language = new Language((int)Configuration::get('PS_LANG_DEFAULT'));

			// Needed for WYSIWYG
			$helper->tpl_vars = array(
				'base_url' => $this->context->shop->getBaseURL(),
				'language' => array(
					'id_lang' => $language->id,
					'iso_code' => $language->iso_code
				),
				'fields_value' => $this->getAddFieldsValues(),
				'languages' => $this->context->controller->getLanguages(),
				'id_language' => $this->context->language->id,
				'image_baseurl' => $this->_path.'images/',
				'exception_list' => ($this->is_version16)?
										$this->displayModuleExceptionList16(unserialize(Configuration::get($this->short_name.'_file_list')), 0):
										$this->displayModuleExceptionList15(unserialize(Configuration::get($this->short_name.'_file_list')), 0),
				'name_controller' => 'MarketingPopup'
			);
		}
		else
		{
			// 1.4 : .tpl du backoffice du module
			if (Tools::getValue('current_tab') == '')
				$current_tab = 'page-1';
			else
				$current_tab = Tools::getValue('current_tab');

			$languages = Language::getLanguages(false);
			$language = new Language((int)Configuration::get('PS_LANG_DEFAULT'));
			$iso_tiny_mce = (file_exists(_PS_ROOT_DIR_.'/js/tiny_mce/langs/'.$language->iso_code.'.js') ? $language->iso_code : 'en');
			$ad = dirname($_SERVER['PHP_SELF']);

			$exception_list = unserialize(Configuration::get($this->short_name.'_file_list'));
			$exception_list = implode(', ', $exception_list);

			// Envoi des paramètres au template
			$this->context->smarty->assign(
					array(
						'js_dir' => _PS_JS_DIR_,
						'css_dir' => _PS_CSS_DIR_,
						'theme_css_dir' => _THEME_CSS_DIR_,
						'isoTinyMCE' => $iso_tiny_mce,
						'ad' => $ad,
						'defaultLanguage' => $language->id,
						'languages' => $languages,
						'current_tab' => $current_tab,
						'employee_cookie' => $this->context->cookie,
						'action' => Tools::safeOutput($_SERVER['REQUEST_URI']),
						'popup_content' => unserialize(rawurldecode(Configuration::get($this->short_name.'_popup_content'))),
						'mp_active' => Configuration::get($this->short_name.'_active'),
						'date_from' => Configuration::get($this->short_name.'_date_from'),
						'date_to' => Configuration::get($this->short_name.'_date_to'),
						'auto_size' => Configuration::get($this->short_name.'_auto_size'),
						'popup_width' => Configuration::get($this->short_name.'_popup_width'),
						'popup_height' => Configuration::get($this->short_name.'_popup_height'),
						'popup_padding' => Configuration::get($this->short_name.'_popup_padding'),
						'popup_margin' => Configuration::get($this->short_name.'_popup_margin'),
						'disp_close' => Configuration::get($this->short_name.'_disp_close'),
						'background_color' => Configuration::get($this->short_name.'_background_color'),
						'popup_delay' => Configuration::get($this->short_name.'_popup_delay'),
						'cookie_expiration' => Configuration::get($this->short_name.'_cookie_expiration'),
						'unit_expiration' => Configuration::get($this->short_name.'_unit_expiration'),
						'exception_list' => $exception_list,
						'mp_tpl_dir' => _PS_MODULE_DIR_.'marketingpopup/views/templates/admin/'
					)
			);
		}

		// Envoi des paramètres au template
		$this->context->smarty->assign(
				array(
					'path' => _MODULE_DIR_.$this->name.'/',
					'iso' => Language::getIsoById($this->context->cookie->id_lang),
					'display_name' => $this->displayName,
					'version' => $this->version,
					'author' => $this->author,
					'contact' => $this->contact,
					'psversion16' => $this->is_version16,
					'mp_social' => Configuration::get($this->short_name.'_mp_social'),
					'mp_facebook' => Configuration::get($this->short_name.'_mp_facebook'),
					'mp_twitter' => Configuration::get($this->short_name.'_mp_twitter'),
					'mp_rss' => Configuration::get($this->short_name.'_mp_rss'),
					'mp_youtube' => Configuration::get($this->short_name.'_mp_youtube'),
					'mp_google' => Configuration::get($this->short_name.'_mp_google'),
					'mp_pinterest' => Configuration::get($this->short_name.'_mp_pinterest'),
					'mp_newsletter' => Configuration::get($this->short_name.'_mp_newsletter')
				)
		);

		if ($this->is_version15)
			return $helper->generateForm($fields_form);
		else
			return $this->display(__FILE__, 'views/templates/admin/admin.tpl');
	}

	public function renderSupportForm()
	{
		// Envoi des paramètres au template
		$this->context->smarty->assign(
				array(
					'path' => _MODULE_DIR_.$this->name.'/',
					'iso' => Language::getIsoById($this->context->cookie->id_lang),
					'display_name' => $this->displayName,
					'version' => $this->version,
					'author' => $this->author,
					'contact' => $this->contact,
					'back_link' => AdminController::$currentIndex.'&configure='.$this->name.'&token='.Tools::getAdminTokenLite('AdminModules'),
					'psversion16' => $this->checkPSVersion('1.6.0.0')
				)
		);

		return $this->display(__FILE__, 'views/templates/admin/support.tpl');
	}

	/**
	 * Méthode hookDisplayMarketingPopupForm()
	 * 
	 * Ajoute des champs personnalisés au formulaire d'administration du module
	 */
	public function hookDisplayMarketingPopupForm()
	{
		// Envoi des paramètres au template
		$this->context->smarty->assign(
				array(
					'cookie_expiration' => Configuration::get($this->short_name.'_cookie_expiration'),
					'unit_expiration' => Configuration::get($this->short_name.'_unit_expiration'),
					'psversion16' => $this->checkPSVersion('1.6.0.0')
				)
		);

		return $this->display(__FILE__, 'views/templates/admin/custom_fields.tpl');
	}

	public function getAddFieldsValues()
	{
		$fields = array();
		$languages = Language::getLanguages(false);

		$fields['mp_active'] = Tools::getValue('active', Configuration::get($this->short_name.'_active'));
		$fields['auto_size'] = Tools::getValue('auto_size', Configuration::get($this->short_name.'_auto_size'));
		$fields['date_from'] = Tools::getValue('date_from', Configuration::get($this->short_name.'_date_from'));
		$fields['date_to'] = Tools::getValue('date_to', Configuration::get($this->short_name.'_date_to'));
		$fields['popup_width'] = Tools::getValue('popup_width', Configuration::get($this->short_name.'_popup_width'));
		$fields['popup_height'] = Tools::getValue('popup_height', Configuration::get($this->short_name.'_popup_height'));
		$fields['popup_padding'] = Tools::getValue('popup_padding', Configuration::get($this->short_name.'_popup_padding'));
		$fields['popup_margin'] = Tools::getValue('popup_margin', Configuration::get($this->short_name.'_popup_margin'));
		$fields['disp_close'] = Tools::getValue('disp_close', Configuration::get($this->short_name.'_disp_close'));
		$fields['background_color'] = Tools::getValue('background_color', Configuration::get($this->short_name.'_background_color'));
		$fields['popup_delay'] = Tools::getValue('popup_delay', Configuration::get($this->short_name.'_popup_delay'));

		$popup_content = unserialize(rawurldecode(Configuration::get($this->short_name.'_popup_content')));
		foreach ($languages as $lang)
			$fields['popup_content'][$lang['id_lang']] = isset($popup_content[$lang['id_lang']])?$popup_content[$lang['id_lang']]:'';

		return $fields;
	}

	/**
	 * Méthode displayModuleExceptionList16()
	 * 
	 * Génère le formulaire d'exceptions de pages pour la version 1.6
	 */
	private function displayModuleExceptionList16($file_list, $shop_id)
	{
		if (!is_array($file_list))
			$file_list = ($file_list) ? array($file_list) : array();

		$content = '<p><input type="text" name="exceptions['.$shop_id.']" value="'.implode(', ', $file_list).'" id="em_text_'.$shop_id.'"/></p>';

		if ($shop_id)
		{
			$shop = new Shop($shop_id);
			$content .= ' ('.$shop->name.')';
		}

		$content .= '<p>
					<select size="25" id="em_list_'.$shop_id.'" multiple="multiple">
					<option disabled="disabled">'.$this->l('___________ CUSTOM ___________').'</option>';

		$controllers = Dispatcher::getControllers(_PS_FRONT_CONTROLLER_DIR_);
		ksort($controllers);

		foreach ($file_list as $k => $v)
			if (!array_key_exists($v, $controllers))
				$content .= '<option value="'.$v.'">'.$v.'</option>';

		$content .= '<option disabled="disabled">'.$this->l('____________ CORE ____________').'</option>';

		foreach ($controllers as $k => $v)
			$content .= '<option value="'.$k.'">'.$k.'</option>';

		if ($this->checkPSVersion('1.6.0.5') && method_exists(Dispatcher, 'getModuleControllers'))
		{
			$modules_controllers_type = array('admin' => $this->l('Admin modules controller'), 'front' => $this->l('Front modules controller'));
			foreach ($modules_controllers_type as $type => $label)
			{
				$content .= '<option disabled="disabled">'.$this->l('____________ '.$label.' ____________').'</option>';
				$all_modules_controllers = Dispatcher::getModuleControllers($type);
				foreach ($all_modules_controllers as $module => $modules_controllers)
					foreach ($modules_controllers as $cont)
						$content .= '<option value="module-'.$module.'-'.$cont.'">module-'.$module.'-'.$cont.'</option>';
			}
		}

		$content .= '</select>
					</p>';

		return $content;
	}

	/**
	 * Méthode displayModuleExceptionList15()
	 * 
	 * Génère le formulaire d'exceptions de pages pour la version 1.5
	 */
	private function displayModuleExceptionList15($file_list, $shop_id)
	{
		if (!is_array($file_list))
			$file_list = ($file_list) ? array($file_list) : array();

		$content = '<input type="text" name="exceptions['.$shop_id.']" size="40" value="'.implode(', ', $file_list).'" id="em_text_'.$shop_id.'">';
		if ($shop_id)
		{
			$shop = new Shop($shop_id);
			$content .= ' ('.$shop->name.')';
		}
		$content .= '<br /><select id="em_list_'.$shop_id.'">';

		$controllers = Dispatcher::getControllers(_PS_FRONT_CONTROLLER_DIR_);
		ksort($controllers);
		foreach ($controllers as $k => $v)
			$content .= '<option value="'.$k.'">'.$k.'</option>';

		$content .= '</select> <input type="button" class="button" value="'.$this->l('Add').'" onclick="position_exception_add('.$shop_id.')" />
				<input type="button" class="button" value="'.$this->l('Remove').'" onclick="position_exception_remove('.$shop_id.')" /><br /><br />';

		return $content;
	}

	/**
	 * Méthode generateCSS()
	 * 
	 * Génère un fichier CSS en fonction des paramètres définits dans le backoffice du module
	 */
	private function generateCSS()
	{
		foreach (Shop::getShops() as $shop)
		{
			// Récupération des paramètres
			$background_color = Configuration::get($this->short_name.'_background_color', null, null, $shop['id_shop']);
			$width = Configuration::get($this->short_name.'_popup_width', null, null, $shop['id_shop']);
			if ($width < 400)
				$input_width = 50;
			else
				$input_width = 70;

			if ($this->checkPSVersion('1.4.9'))
				$source_css_file = 'static.css';
			else
				$source_css_file = 'static_1.4.css';

			// Récupération du fichier CSS static
			$css_content = Tools::file_get_contents(dirname(__FILE__).'/css/'.$source_css_file);

			// Génération du fichier CSS
			$css_content .= '
	.marketing-popup-bg,
	#fancybox-outer {
		background-color: '.$background_color.' !important;
	}
	#marketing-popup .block-newsletter .newsletter-input {
		width: '.$input_width.'%;
	}';

			if (!$this->checkPSVersion('1.6.0.0') && $this->checkPSVersion())
			{
				$css_content .= '
	#marketing-popup .block-newsletter,
	#marketing-popup .block-social {
		line-height:28px;
	}
	#marketing-popup .block-newsletter h4,
	#marketing-popup .block-social h4 {
		line-height:37px;
	}';
			}
			else if (!$this->checkPSVersion())
			{
				$css_content .= '
	#marketing-popup .block-newsletter .newsletter-input {
		width:60%;
	}
	#marketing-popup .block-newsletter,
	#marketing-popup .block-social {
		line-height:18px;
	}
	#marketing-popup .block-newsletter h4,
	#marketing-popup .block-social h4 {
		line-height:37px;
	}';
			}

			file_put_contents(dirname(__FILE__).'/css/marketingpopup-s-'.$shop['id_shop'].'.css', $css_content, LOCK_EX);
		}
	}

	/**
	 * Méthode checkPSVersion()
	 * 
	 * Compare la version de Prestashop passée en paramètre avec la version courante
	 * 
	 * @param string $version Version à comparer
	 * @param string $compare Sens de la comparaison
	 * 
	 * @return boolean True si la comparaison est vérifiée
	 * 
	 */
	private function checkPSVersion($version = '1.5.0.0', $compare = '>')
	{
		return version_compare(_PS_VERSION_, $version, $compare);
	}

	private function checkPopupDisplay()
	{
		$display = true;
		$active = Configuration::get($this->short_name.'_active');
		$date_from = Configuration::get($this->short_name.'_date_from');
		$date_to = Configuration::get($this->short_name.'_date_to');

		// Test si la popup est active et si on est bien entre les dates d'affichage
		if ($active && time() > strtotime($date_from) && time() < strtotime($date_to))
		{
			// Test l'existence du cookie
			if (isset($_COOKIE['marketing_popup-s-'.$this->context->shop->id]) && $_COOKIE['marketing_popup-s-'.$this->context->shop->id] == 1)
				$display = false;

			// Test si l'inscription à la newsletter a été soumis
			if (Tools::isSubmit('submitNewsletter'))
				$display = false;

			// Test si la page courante fait partie des exceptions de pages
			$exceptions = unserialize(Configuration::get($this->short_name.'_file_list'));
			$php_self = str_replace('-', '', $this->context->controller->php_self);
			if ($this->checkPSVersion())
			{
				if (in_array($php_self, $exceptions))
					$display = false;
			}
			else
			{
				if (in_array(basename($_SERVER['PHP_SELF']), $exceptions))
					$display = false;
			}
		}
		else
			$display = false;

		return $display;
	}

	private function createCookie()
	{
		$unit_expiration = Configuration::get($this->short_name.'_unit_expiration');
		$cookie_expiration = Configuration::get($this->short_name.'_cookie_expiration');

		switch ($unit_expiration)
		{
			// Unité : Jours
			case 0:
				$expiration_time = $cookie_expiration * 60 * 60 * 24;
				break;
			// Unité : Heures
			case 1:
				$expiration_time = $cookie_expiration * 60 * 60;
				break;
			// Unité : Minutes
			case 2:
				$expiration_time = $cookie_expiration * 60;
				break;
			// Unité : Secondes
			case 3:
				$expiration_time = $cookie_expiration;
				break;
			default:
				$expiration_time = 0;
				break;
		}

		setcookie('marketing_popup-s-'.$this->context->shop->id, 1, time() + $expiration_time);
	}

	/**
	 * Méthode hookHeader()
	 * 
	 * Ajoute des fichiers css et/ou js dans la balise <head> de la page Html
	 */
	public function hookHeader()
	{
		if (!$this->checkPopupDisplay())
			return;

		if ($this->checkPSVersion('1.4.9'))
		{
			$this->context->controller->addCSS($this->_path.'css/jquery.fancybox.2.1.5.css');
			$this->context->controller->addJS($this->_path.'js/jquery.fancybox.2.1.5.js');
			$this->context->controller->addJS($this->_path.'js/marketingpopup.js');
		}
		else
		{
			$this->context->controller->addCSS($this->_path.'css/jquery.fancybox-1.3.4.css');
			$this->context->controller->addJS($this->_path.'js/jquery.fancybox-1.3.4.pack.js');
			$this->context->controller->addJS($this->_path.'js/marketingpopup_1.4.js');
		}

		$this->context->controller->addCSS($this->_path.'css/marketingpopup-s-'.$this->context->shop->id.'.css');

		if (!$this->checkPSVersion('1.6.0.0'))
			$this->context->controller->addCSS($this->_path.'css/font-awesome.css');
	}

	/**
	 * Méthode prepareHook()
	 * 
	 * Initialise et envoi toutes les variables à la vue du module
	 */
	private function prepareHook()
	{
		// Création du Cookie
		$this->createCookie();

		// Récupération des paramètres du module
		$popup_content = unserialize(rawurldecode(Configuration::get($this->short_name.'_popup_content')));
		$popup_content_single = $popup_content[$this->context->cookie->id_lang];

		$popup_width = Configuration::get($this->short_name.'_popup_width');
		$popup_height = Configuration::get($this->short_name.'_popup_height');

		// Envoi des paramètres au template
		$this->context->smarty->assign(
				array(
					$this->short_name.'_path' => _MODULE_DIR_.$this->name.'/',
					$this->short_name.'_popup_content' => $popup_content_single,
					$this->short_name.'_auto_size' => Configuration::get($this->short_name.'_auto_size'),
					$this->short_name.'_popup_width' => ($popup_width != '' ? $popup_width : 0),
					$this->short_name.'_popup_height' => ($popup_height != '' ? $popup_height : 0),
					$this->short_name.'_popup_padding' => Configuration::get($this->short_name.'_popup_padding'),
					$this->short_name.'_popup_margin' => Configuration::get($this->short_name.'_popup_margin'),
					$this->short_name.'_disp_close' => Configuration::get($this->short_name.'_disp_close'),
					$this->short_name.'_popup_delay' => Configuration::get($this->short_name.'_popup_delay'),
					$this->short_name.'_social' => Configuration::get($this->short_name.'_mp_social'),
					$this->short_name.'_facebook' => Configuration::get($this->short_name.'_mp_facebook'),
					$this->short_name.'_twitter' => Configuration::get($this->short_name.'_mp_twitter'),
					$this->short_name.'_rss' => Configuration::get($this->short_name.'_mp_rss'),
					$this->short_name.'_youtube' => Configuration::get($this->short_name.'_mp_youtube'),
					$this->short_name.'_google' => Configuration::get($this->short_name.'_mp_google'),
					$this->short_name.'_pinterest' => Configuration::get($this->short_name.'_mp_pinterest'),
					$this->short_name.'_newsletter' => Configuration::get($this->short_name.'_mp_newsletter'),
					$this->short_name.'_psversion15' => $this->checkPSVersion(),
					$this->short_name.'_psversion16' => $this->checkPSVersion('1.6.0.0')
				)
		);
	}

	/**
	 * Méthode hookTop()
	 * 
	 * Affiche le module en haut du site
	 */
	public function hookTop()
	{
		if (!$this->checkPopupDisplay())
			return;

		$this->prepareHook();

		if (!$this->checkPSVersion('1.4.9'))
			return $this->display(__FILE__, 'views/templates/hook/marketingpopup_1.4.tpl');
		else
			return $this->display(__FILE__, 'views/templates/hook/marketingpopup.tpl');
	}
}