<?php
/**
* WhatsApp Chat - Live chat
*
* NOTICE OF LICENSE
*
* This product is licensed for one customer to use on one installation (test stores and multishop included).
* Site developer has the right to modify this module to suit their needs, but can not redistribute the module in
* whole or in part. Any other use of this module constitues a violation of the user agreement.
*
* DISCLAIMER
*
* NO WARRANTIES OF DATA SAFETY OR MODULE SECURITY
* ARE EXPRESSED OR IMPLIED. USE THIS MODULE IN ACCORDANCE
* WITH YOUR MERCHANT AGREEMENT, KNOWING THAT VIOLATIONS OF
* PCI COMPLIANCY OR A DATA BREACH CAN COST THOUSANDS OF DOLLARS
* IN FINES AND DAMAGE A STORES REPUTATION. USE AT YOUR OWN RISK.
*
*  @author    idnovate.com <info@idnovate.com>
*  @copyright 2017 idnovate.com
*  @license   See above
*/

include_once(_PS_MODULE_DIR_.'whatsappchat/classes/WhatsappChatBlock.php');

class WhatsAppChat extends Module
{
    public function __construct()
    {
        $this->name = 'whatsappchat';
        $this->tab = 'front_office_features';
        $this->version = '1.1.5';
        $this->author = 'idnovate';
        $this->bootstrap = true;
        $this->module_key = 'fb00ab599d53a30abdaeae23c95fc2a7';
        $this->addons_id_product = '26395';

        parent::__construct();

        $this->displayName = $this->l('WhatsApp Chat - Live chat with your customers');
        $this->description = $this->l('Chat with your customers through WhatsApp, the most popular messaging app');
        $this->confirmUninstall = $this->l('Are you sure you want to uninstall module?');

        /* Backward compatibility */
        if (version_compare(_PS_VERSION_, '1.5', '<')) {
            require(_PS_MODULE_DIR_.$this->name.'/backward_compatibility/backward.php');
            $this->local_path = _PS_MODULE_DIR_.$this->name.'/';
        }

        /* Mobile detect library */
        if (version_compare(_PS_VERSION_, '1.5', '<')) {
            include_once(_PS_MODULE_DIR_.$this->name.'/backward_compatibility/Mobile_Detect.php');
        } else {
            include_once(_PS_TOOL_DIR_.'mobile_Detect/Mobile_Detect.php');
        }

        $this->warning = $this->getWarnings(false, true);
    }

    public function install()
    {
        $result = true;

        $result &= parent::install()
            && $this->createDb()
            && $this->registerHook('header')
            && $this->registerHook('footer')
            && $this->registerHook('leftColumn')
            && $this->registerHook('rightColumn')
            && $this->registerHook('top')
            && $this->registerHook('home')
            && $this->registerHook('shoppingCart')
            && $this->registerHook('shoppingCartExtra')
            && $this->registerHook('paymentTop')
            && $this->registerHook('beforeCarrier')
            && $this->registerHook('customerAccount')
            && $this->registerHook('myAccountBlock')
            && $this->registerHook('orderConfirmation')
            && $this->registerHook('backOfficeHeader');

        if (version_compare(_PS_VERSION_, '1.5', '>=')) {
            $result &= $this->installTabs()
                && $this->registerHook('displayBanner')
                && $this->registerHook('displayTopColumn')
                && $this->registerHook('displayNav')
                && $this->registerHook('displayproductButtons')
                && $this->registerHook('displayLeftColumnProduct')
                && $this->registerHook('displayRightColumnProduct')
                && $this->registerHook('displayFooterProduct')
                && $this->registerHook('displayShoppingCartFooter')
                && $this->registerHook('displayCustomerAccountForm')
                && $this->registerHook('displayCustomerAccountFormTop')
                && $this->registerHook('displayCustomerIdentityForm')
                && $this->registerHook('displayMaintenance');
        } else {
            $result &= $this->registerHook('extraLeft')
                && $this->registerHook('extraRight')
                && $this->registerHook('productActions')
                && $this->registerHook('productfooter');
        }

        return $result;
    }

    public function uninstall()
    {
        return parent::uninstall()
            && $this->uninstallDb()
            && Configuration::deleteByName('WA_CHAT_MOBILE')
            && Configuration::deleteByName('WA_CHAT_MESSAGE')
            && Configuration::deleteByName('WA_BADGE_MESSAGE')
            && Configuration::deleteByName('WA_BADGE_POSITION');
    }

    public function createDb()
    {
        $result = true;

        $result &= Db::getInstance()->execute(
            'CREATE TABLE IF NOT EXISTS `'._DB_PREFIX_.'whatsappchatblock` (
              `id_whatsappchatblock` int(11) NOT NULL AUTO_INCREMENT,
              `id_shop` int(10) NOT NULL,
              `id_hook` varchar(150) NOT NULL,
              `open_chat` tinyint(1) NOT NULL,
              `position` varchar(150) NOT NULL,
              `color` varchar(32) NULL DEFAULT "",
              `active` int(1) NOT NULL DEFAULT 0,
              `customer_groups` varchar(60) NULL DEFAULT "all",
              `only_home` int(1) NOT NULL DEFAULT 0,
              `chat_group` varchar(60) NULL DEFAULT "",
              `badge_width` int(3) NULL,
              `only_mobile` int(1) NOT NULL DEFAULT 0,
              PRIMARY KEY (`id_whatsappchatblock`),
              KEY `id_shop_id_hook` (`id_shop`,`id_hook`)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8;'
        );

        $result &= Db::getInstance()->execute(
            'CREATE TABLE IF NOT EXISTS `'._DB_PREFIX_.'whatsappchatblock_lang` (
              `id_whatsappchatblock` int(11) NOT NULL,
              `id_lang` int(11) NOT NULL,
              `message` text,
              `mobile_phone` varchar(32) NULL DEFAULT "",
              KEY `id_whatsappchatblock_id_lang` (`id_whatsappchatblock`,`id_lang`)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8;'
        );

        return $result;
    }

    public function uninstallDb()
    {
        $result = true;

        $result &= Db::getInstance()->Execute('DROP TABLE IF EXISTS `'._DB_PREFIX_.'whatsappchatblock`;');
        $result &= Db::getInstance()->Execute('DROP TABLE IF EXISTS `'._DB_PREFIX_.'whatsappchatblock_lang`;');

        return $result;
    }

    public function installTabs()
    {
        $moduleTab = array(
            'tabClassName' => 'AdminWhatsappChat',
            'tabParentName' => '',
            'tabName' => $this->l('WhatsApp Chat'),
        );

        $id_tab = Tab::getIdFromClassName($moduleTab['tabClassName']);
        if (!$id_tab) {
            $tab = new Tab();
            $tab->class_name = $moduleTab['tabClassName'];
            if ($moduleTab['tabParentName']) {
                $tab->id_parent = Tab::getIdFromClassName($moduleTab['tabParentName']);
            } else {
                $tab->id_parent = -1;
            }

            $tab->module = $this->name;

            $languages = Language::getLanguages(false);
            foreach ($languages as $lang) {
                $tab->name[$lang['id_lang']] = $moduleTab['tabName'];
            }

            $tab->add();
        }

        return true;
    }

    /* All pages */
    public function hookHeader()
    {
        $this->context->controller->addCSS($this->_path.'views/css/whatsapp.css', 'all');

        $html = $this->displayBlock('badge');
        $html .= $this->displayBlock('topWidth');
        $html .= $this->displayBlock('bottomWidth');
        return $html.$this->displayBlock('header');
    }

    public function hookFooter()
    {
        return $this->displayBlock('footer');
    }

    public function hookDisplayBanner()
    {
        return $this->displayBlock(__FUNCTION__);
    }

    public function hookLeftColumn()
    {
        return $this->displayBlock('leftcolumn');
    }

    public function hookRightColumn()
    {
        return $this->displayBlock('rightcolumn');
    }

    public function hookDisplayTopColumn()
    {
        return $this->displayBlock(__FUNCTION__);
    }

    public function hookTop()
    {
        return $this->displayBlock('top');
    }

    /* Homepage */
    public function hookDisplayNav()
    {
        return $this->displayBlock(__FUNCTION__);
    }

    public function hookHome()
    {
        return $this->displayBlock('home');
    }

    /* Product page */
    public function hookDisplayProductButtons()
    {
        return $this->displayBlock(__FUNCTION__);
    }

    public function hookDisplayLeftColumnProduct()
    {
        return $this->displayBlock(__FUNCTION__);
    }

    public function hookDisplayRightColumnProduct()
    {
        return $this->displayBlock(__FUNCTION__);
    }

    public function hookDisplayFooterProduct()
    {
        return $this->displayBlock(__FUNCTION__);
    }

    /* Checkout */
    public function hookShoppingCart()
    {
        return $this->displayBlock('shoppingcart');
    }

    public function hookShoppingCartExtra()
    {
        return $this->displayBlock('shoppingcartextra');
    }

    public function hookPaymentTop()
    {
        return $this->displayBlock('paymenttop');
    }

    public function hookBeforeCarrier()
    {
        return $this->displayBlock('beforecarrier');
    }

    /* Order */
    public function hookOrderConfirmation()
    {
        return $this->displayBlock('orderconfirmation');
    }

    /* Customer account */
    public function hookCustomerAccount()
    {
        return $this->displayBlock('customeraccount');
    }

    public function hookDisplayCustomerAccountForm()
    {
        return $this->displayBlock(__FUNCTION__);
    }

    public function hookDisplayCustomerAccountFormTop()
    {
        return $this->displayBlock(__FUNCTION__);
    }

    public function hookDisplayCustomerIdentityForm()
    {
        return $this->displayBlock(__FUNCTION__);
    }

    public function hookMyAccountBlock()
    {
        return $this->displayBlock('myaccountblock');
    }

     /* Maintenance */
    public function hookDisplayMaintenance()
    {
        $html = '<link href="'.$this->_path.'views/css/whatsapp.css" rel="stylesheet">';
        return $html.$this->displayBlock(__FUNCTION__);
    }

    /* 1.4 */
    public function hookExtraLeft()
    {
        return $this->displayBlock('extraleft');
    }
    public function hookExtraRight()
    {
        return $this->displayBlock('extraright');
    }
    public function hookProductActions()
    {
        return $this->displayBlock('productactions');
    }
    public function hookProductFooter()
    {
        return $this->displayBlock('productfooter');
    }

    public function displayBlock($hook = false, $from_bo = false, $id_whatsappchatblock = false)
    {
        if (version_compare(_PS_VERSION_, '1.5', '<')) {
            if (!($mobileNumber = Configuration::get('WA_CHAT_MOBILE'))) {
                return $mobileNumber;
            }
        }
        $only_active = $from_bo ? false: true;
        if (version_compare(_PS_VERSION_, '1.5', '>=')) {
            $whatsappchatblock = new WhatsappChatBlock();
            if ($message = $whatsappchatblock->getWhatsappChatByHook($hook, $only_active, $from_bo, $id_whatsappchatblock)) {
                if ($message['mobile_phone'] == '' && $message['message'] == '') {
                    return false;
                }
	            $mobile = new Mobile_Detect();
                if ($message['only_mobile'] == '1' && !$mobile->isMobile() && $from_bo == false) {
                    return false;
                }
                if ($message['only_home'] == '1' && Dispatcher::getInstance()->getController() != 'index' && $from_bo == false) {
                    return false;
                }
                if (!$from_bo && $customer_groups = $this->context->customer->getGroupsStatic($this->context->customer->id)) {
                    $filter_groups = explode(';', $message['customer_groups']);
                    if (count(array_intersect($filter_groups, $customer_groups)) == 0 && $message['customer_groups'] != 'all') {
                        return false;
                    }
                }
                $url = $this->getWhatsappUrl($message['mobile_phone'], Configuration::get('WA_CHAT_MESSAGE', $this->context->cookie->id_lang), $message['chat_group']);
                $this->context->smarty->assign(array(
                    'font_awesome'  => ($hook == 'hookDisplayMaintenance' ? false : Configuration::get('WA_FONT_AWESOME')),
                    'message'       => $message['message'],
                    'whatsapp_class'=> $hook,
                    'position'      => $message['position'],
                    'open_chat'     => $message['open_chat'],
                    'mobile_phone'  => $message['mobile_phone'],
                    'color'         => $message['color'],
                    'url'           => $url,
                    'from_bo'       => ($from_bo ? '1' : '0')
                ));
                return $this->display(__FILE__, 'template.tpl');
            }
        } else {
            if (Configuration::get('WA_'.Tools::strtoupper($hook).'_ENABLE')) {
                $url = $this->getWhatsappUrl(Configuration::get('WA_CHAT_MOBILE'), Configuration::get('WA_CHAT_MESSAGE', $this->context->cookie->id_lang));
                $this->context->smarty->assign(array(
                    'font_awesome'  => false,
                    'message'       => Configuration::get('WA_'.Tools::strtoupper($hook).'_MESSAGE', (int)$this->context->cookie->id_lang),
                    'whatsapp_class'=> $hook,
                    'position'      => Configuration::get('WA_'.Tools::strtoupper($hook).'_POSITION'),
                    'open_chat'     => Configuration::get('WA_'.Tools::strtoupper($hook).'_CHAT'),
                    'url'           => $url,
                    'from_bo'       => ($from_bo ? '1' : '0')
                ));
                return $this->display(__FILE__, 'views/templates/hook/template.tpl');
            }
        }
    }

    public function hookbackOfficeHeader()
    {
        if (version_compare(_PS_VERSION_, '1.5', '<')) {
            return true;
        }
        if (Module::isEnabled('whatsappchat') &&
            ((version_compare(_PS_VERSION_, '1.5', '>=') && Tools::strtolower(Dispatcher::getInstance()->getController()) == 'admincustomers') ||
                (version_compare(_PS_VERSION_, '1.5', '<') && Tools::strtolower(Tools::getValue('tab')) == 'admincustomers') ||
            Tools::getValue('controller') == 'AdminWhatsappChat' || Tools::getValue('tab') == 'AdminWhatsappChat')
        ) {
            $this->context->controller->addCSS($this->_path.'views/css/whatsapp.css', 'all');
            $this->context->controller->addCSS($this->_path.'views/css/bo_whatsapp.css', 'all');
            $this->context->smarty->assign(array(
                'action' => $this->l('WhatsApp'),
                'action_whatsappchat' => $this->l('WhatsApp with this customer'),
                'admin_base_dir' => $this->currentPageURL(),
                'this_path_bo' => $this->_path,
                'iso_code' => $this->context->language->iso_code,
                'whatsappchat_admincontroller' => $this->context->link->getAdminLink('AdminWhatsappChat', true, $this->context->language->id),
                'url' => false,
                'show_button' => true,
                'token' => Tools::getAdminToken('AdminCustomers')
            ));
            if (Tools::getIsset('viewcustomer') && Tools::getValue('id_customer') > 0) {
                $address_id = Address::getFirstCustomerAddressId((int)Tools::getValue('id_customer'), true);
                if ($address_id > 0) {
                    $address = new Address((int)$address_id);
                    $phone = $address->phone_mobile;
                    if (!Validate::isPhoneNumber($phone) || $phone == '') {
                        $phone = $address->phone;
                        if (!Validate::isPhoneNumber($phone) || $phone == '') {
                            $this->context->smarty->assign(array(
                                'show_button' => false
                            ));
                        }
                    }
                    $phone = $this->formatMobilePhoneForWhatsapp($phone, $address->id_country);
                    $this->context->smarty->assign(array(
                        'url' => $this->getWhatsappUrl($phone)
                    ));
                }
            }
            if (version_compare(_PS_VERSION_, '1.5', '<')) {
                return $this->display(__FILE__, '/views/templates/hook/bo_customers_grid_action.tpl');
            } else {
                return $this->display(__FILE__, 'bo_customers_grid_action.tpl');
            }
        }
    }

    public function getContent()
    {
        if (version_compare(_PS_VERSION_, '1.5', '<')) {
            $html = '';

            if (((bool)Tools::isSubmit('submitwhatsappchatModule')) == true) {
                $html .= $this->postProcess();
            }

            if ($warnings = $this->getWarnings(false)) {
                $html .= $this->displayError($warnings);
            }

            return $html.$this->renderForm14();
        }

        if ((version_compare(_PS_VERSION_, '1.5.0.13', '<') && Module::isInstalled('whatsappchat'))
            || (version_compare(_PS_VERSION_, '1.5.0.13', '>=') && Module::isEnabled('whatsappchat'))) {
            $this->installTabs();
        }

        return Tools::redirectAdmin('index.php?controller=AdminWhatsappChat&token='.Tools::getAdminTokenLite('AdminWhatsappChat'));
    }

    protected function renderForm()
    {
        $html = '';

        $helper = new HelperForm();

        $helper->show_toolbar = false;
        $helper->table = $this->table;
        $helper->module = $this;
        $helper->default_form_language = $this->context->language->id;
        $helper->allow_employee_form_lang = Configuration::get('PS_BO_ALLOW_EMPLOYEE_FORM_LANG', 0);

        $helper->identifier = $this->identifier;
        $helper->submit_action = 'submitwhatsappchatModule';
        $helper->currentIndex = $this->context->link->getAdminLink('AdminModules', false)
            .'&configure='.$this->name.'&tab_module='.$this->tab.'&module_name='.$this->name;
        $helper->token = Tools::getAdminTokenLite('AdminModules');

        $helper->tpl_vars = array(
            'fields_value' => $this->getConfigFormValues(),
            'languages'     => $this->context->controller->getLanguages(),
            'id_language'   => $this->context->language->id,
        );

        $html .= $helper->generateForm($this->getConfigForm());

        if (version_compare(_PS_VERSION_, '1.6', '>=')) {
            $this->context->smarty->assign(array(
                'this_path'     => $this->_path,
                'support_id'    => $this->addons_id_product
            ));
            $available_iso_codes = array('en', 'es');
            $default_iso_code = 'en';
            $template_iso_suffix = in_array($this->context->language->iso_code, $available_iso_codes) ? $this->context->language->iso_code : $default_iso_code;
            $html .= $this->context->smarty->fetch($this->local_path.'views/templates/admin/company/information_'.$template_iso_suffix.'.tpl');
        }

        return $html;
    }

    protected function renderForm14()
    {
        $html = '';

        $helper = new Helper();

        $helper->tpl_vars = array(
            'fields_value' => $this->getConfigFormValues(),
            'languages' => Language::getLanguages(false),
            'id_language' => $this->context->language->id,
            'THEME_LANG_DIR' => _PS_IMG_.'l/'
        );

        $html .= $helper->generateForm($this->getConfigForm());

        return $html;
    }

    protected function postProcess()
    {
        $form_values = $this->getConfigFormValues();

        foreach (array_keys($form_values) as $key) {
            Configuration::updateValue($key, $form_values[$key]);
        }

        return $this->displayConfirmation($this->l('Configuration saved successfully.'));
    }

    protected function getConfigForm()
    {
        $fields = array();

        $fields[]['form'] = array(
            'legend' => array(
                'title' => $this->l('General configuration'),
                'icon' => 'icon-cogs',
            ),
            'input' => array(
                array(
                    'type'  => 'text',
                    'label' => $this->l('Mobile phone number'),
                    'name'  => 'WA_CHAT_MOBILE',
                    'desc' => $this->l('Introduce mobile phone number with the international country code, without "+" character.').'<br />'.$this->l('Example: Introduce 341234567 for (+34) 1234567.'),
                    'col'   => 2,
                    'class' => 't',
                ),
                array(
                    'type'  => 'text',
                    'label' => $this->l('Default chat message'),
                    'name'  => 'WA_CHAT_MESSAGE',
                    'lang'  => true,
                    'col'   => 4,
                    'class' => 't',
                ),
            )
        );

        $fields[]['form'] = array(
            'legend' => array(
                'title' => $this->l('Badge configuration'),
                'icon' => 'icon-cogs',
            ),
            'input' => array(
                array(
                    'type' => 'radio',
                    'label' => $this->l('Display badge?'),
                    'name' => 'WA_BADGE_ENABLE',
                    'is_bool' => true,
                    'class' => 't',
                    'values' => array(
                        array(
                            'id' => 'WA_BADGE_ENABLE_on',
                            'value' => 1,
                            'label' => $this->l('Yes')),
                        array(
                            'id' => 'WA_BADGE_ENABLE_off',
                            'value' => 0,
                            'label' => $this->l('No')),
                    ),
                ),
                array(
                    'type'  => 'text',
                    'label' => $this->l('Badge message'),
                    'name'  => 'WA_BADGE_MESSAGE',
                    'lang'  => true,
                    'col'   => 4,
                    'class' => 't',
                ),
                array(
                    'type' => 'select',
                    'label' => $this->l('Badge position'),
                    'name' => 'WA_BADGE_POSITION',
                    'class' => 't',
                    'options' => array(
                        'query' => array(
                            array(
                                'id' => 'bottom-left',
                                'name' => $this->l('Bottom left')
                            ),
                            array(
                                'id' => 'bottom-right',
                                'name' => $this->l('Bottom right'),
                            )
                        ),
                        'id' => 'id',
                        'name' => 'name'
                    ),
                ),
                array(
                    'type' => 'radio',
                    'label' => $this->l('Open chat in a new window?'),
                    'name' => 'WA_BADGE_CHAT',
                    'is_bool' => true,
                    'class' => 't',
                    'values' => array(
                        array(
                            'id' => 'WA_BADGE_CHAT_on',
                            'value' => 1,
                            'label' => $this->l('Yes')),
                        array(
                            'id' => 'WA_BADGE_CHAT_off',
                            'value' => 0,
                            'label' => $this->l('No')),
                    ),
                ),
            ),
            'submit' => array(
                'title' => $this->l('Save'),
                'type' => 'submit',
                'name' => 'submitwhatsappchatModule',
            ),
        );

        $positions = array(
            array(
                'id' => 'HEADER',
                'name' => 'header'
            ),
            array(
                'id' => 'TOP',
                'name' => 'top'
            ),
            array(
                'id' => 'LEFTCOLUMN',
                'name' => 'left column'
            ),
            array(
                'id' => 'RIGHTCOLUMN',
                'name' => 'right column'
            ),
            array(
                'id' => 'FOOTER',
                'name' => 'footer'
            )
        );

        $sectionFields = array();
        foreach ($positions as $position) {
            $sectionFields = array_merge($sectionFields, array(
                array(
                    'type'          => 'html',
                    'html_content'  => '<hr>',
                ),
                array(
                    'type' => 'radio',
                    'label' => $this->l('Display block in ').$position['name'].'?',
                    'name' => 'WA_'.$position['id'].'_ENABLE',
                    'is_bool' => true,
                    'class' => 't',
                    'values' => array(
                        array(
                            'id' => 'WA_'.$position['id'].'_ENABLE_on',
                            'value' => 1,
                            'label' => $this->l('Yes')),
                        array(
                            'id' => 'WA_'.$position['id'].'_ENABLE_off',
                            'value' => 0,
                            'label' => $this->l('No')),
                    ),
                ),
                array(
                    'type'  => 'text',
                    'label' => $this->l('Block message'),
                    'name'  => 'WA_'.$position['id'].'_MESSAGE',
                    'lang'  => true,
                    'col'   => 4,
                    'class' => 't',
                ),
                array(
                    'type' => 'select',
                    'label' => $this->l('Position'),
                    'name' => 'WA_'.$position['id'].'_POSITION',
                    'class' => 't',
                    'options' => array(
                        'query' => array(
                            array(
                                'id' => 'left',
                                'name' => $this->l('Left')
                            ),
                            array(
                                'id' => 'center',
                                'name' => $this->l('Center'),
                            ),
                            array(
                                'id' => 'right',
                                'name' => $this->l('Right'),
                            )
                        ),
                        'id' => 'id',
                        'name' => 'name'
                    ),
                ),
                array(
                    'type' => 'radio',
                    'label' => $this->l('Open chat in a new window?'),
                    'name' => 'WA_'.$position['id'].'_CHAT',
                    'is_bool' => true,
                    'class' => 't',
                    'values' => array(
                        array(
                            'id' => 'WA_'.$position['id'].'_CHAT_on',
                            'value' => 1,
                            'label' => $this->l('Yes')),
                        array(
                            'id' => 'WA_'.$position['id'].'_CHAT_off',
                            'value' => 0,
                            'label' => $this->l('No')),
                    ),
                )
            ));
        }

        $sectionFields = array_merge(
            array(array(
                'type'  => 'html',
                'html_content'  => '<div style="text-align:center; font-weight:bold"><a target="_blank" href="'.$this->_path.'views/img/allpages_14.png">'.$this->l('View hook position 🔗').'</a></div><br />',
            )),
            $sectionFields
        );

        $fields[]['form'] = array(
            'legend' => array(
                'title' => $this->l('All pages configuration'),
                'icon' => 'icon-cogs',
            ),
            'input' => $sectionFields,
            'submit' => array(
                'title' => $this->l('Save'),
                'type' => 'submit',
                'name' => 'submitwhatsappchatModule',
            ),
        );

        $fields[]['form'] = array(
            'legend' => array(
                'title' => $this->l('Home configuration'),
                'icon' => 'icon-cogs',
            ),
            'input' => array(
                array(
                    'type'  => 'html',
                    'html_content'  => '<div style="text-align:center; font-weight:bold"><a target="_blank" href="'.$this->_path.'views/img/home_14.png">'.$this->l('View hook position 🔗').'</a></div><br />',
                ),
                array(
                    'type' => 'radio',
                    'label' => $this->l('Display block in home?'),
                    'name' => 'WA_HOME_ENABLE',
                    'is_bool' => true,
                    'class' => 't',
                    'values' => array(
                        array(
                            'id' => 'WA_HOME_ENABLE_on',
                            'value' => 1,
                            'label' => $this->l('Yes')),
                        array(
                            'id' => 'WA_HOME_ENABLE_off',
                            'value' => 0,
                            'label' => $this->l('No')),
                    ),
                ),
                array(
                    'type'  => 'text',
                    'label' => $this->l('Block message'),
                    'name'  => 'WA_HOME_MESSAGE',
                    'lang'  => true,
                    'col'   => 4,
                    'class' => 't',
                ),
                array(
                    'type' => 'select',
                    'label' => $this->l('Position'),
                    'name' => 'WA_HOME_POSITION',
                    'class' => 't',
                    'options' => array(
                        'query' => array(
                            array(
                                'id' => 'left',
                                'name' => $this->l('Left')
                            ),
                            array(
                                'id' => 'center',
                                'name' => $this->l('Center'),
                            ),
                            array(
                                'id' => 'right',
                                'name' => $this->l('Right'),
                            )
                        ),
                        'id' => 'id',
                        'name' => 'name'
                    ),
                ),
                array(
                    'type' => 'radio',
                    'label' => $this->l('Open chat in a new window?'),
                    'name' => 'WA_HOME_CHAT',
                    'is_bool' => true,
                    'class' => 't',
                    'values' => array(
                        array(
                            'id' => 'WA_HOME_CHAT_on',
                            'value' => 1,
                            'label' => $this->l('Yes')),
                        array(
                            'id' => 'WA_HOME_CHAT_off',
                            'value' => 0,
                            'label' => $this->l('No')),
                    ),
                ),
            ),
            'submit' => array(
                'title' => $this->l('Save'),
                'type' => 'submit',
                'name' => 'submitwhatsappchatModule',
            ),
        );

        $positions = array(
            array(
                'id' => 'EXTRALEFT',
                'name' => 'left column'
            ),
            array(
                'id' => 'EXTRARIGHT',
                'name' => 'right column'
            ),
            array(
                'id' => 'PRODUCTACTIONS',
                'name' => 'product actions'
            ),
            array(
                'id' => 'PRODUCTFOOTER',
                'name' => 'product footer'
            ),
        );

        $sectionFields = array();
        foreach ($positions as $position) {
            $sectionFields = array_merge($sectionFields, array(
                array(
                    'type' => 'radio',
                    'label' => $this->l('Display block in ').$position['name'].'?',
                    'name' => 'WA_'.$position['id'].'_ENABLE',
                    'is_bool' => true,
                    'class' => 't',
                    'values' => array(
                        array(
                            'id' => 'WA_'.$position['id'].'_ENABLE_on',
                            'value' => 1,
                            'label' => $this->l('Yes')),
                        array(
                            'id' => 'WA_'.$position['id'].'_ENABLE_off',
                            'value' => 0,
                            'label' => $this->l('No')),
                    ),
                ),
                array(
                    'type'  => 'text',
                    'label' => $this->l('Block message'),
                    'name'  => 'WA_'.$position['id'].'_MESSAGE',
                    'lang'  => true,
                    'col'   => 4,
                    'class' => 't',
                ),
                array(
                    'type' => 'select',
                    'label' => $this->l('Position'),
                    'name' => 'WA_'.$position['id'].'_POSITION',
                    'class' => 't',
                    'options' => array(
                        'query' => array(
                            array(
                                'id' => 'left',
                                'name' => $this->l('Left')
                            ),
                            array(
                                'id' => 'center',
                                'name' => $this->l('Center'),
                            ),
                            array(
                                'id' => 'right',
                                'name' => $this->l('Right'),
                            )
                        ),
                        'id' => 'id',
                        'name' => 'name'
                    ),
                ),
                array(
                    'type' => 'radio',
                    'label' => $this->l('Open chat in a new window?'),
                    'name' => 'WA_'.$position['id'].'_CHAT',
                    'is_bool' => true,
                    'class' => 't',
                    'values' => array(
                        array(
                            'id' => 'WA_'.$position['id'].'_CHAT_on',
                            'value' => 1,
                            'label' => $this->l('Yes')),
                        array(
                            'id' => 'WA_'.$position['id'].'_CHAT_off',
                            'value' => 0,
                            'label' => $this->l('No')),
                    ),
                ),
                array(
                    'type'          => 'html',
                    'html_content'  => '<hr>',
                ),
            ));
        }

        $sectionFields = array_merge(
            array(array(
                'type'  => 'html',
                'html_content'  => '<div style="text-align:center; font-weight:bold"><a target="_blank" href="'.$this->_path.'views/img/product_14.png">'.$this->l('View hook position 🔗').'</a></div><br />',
            )),
            $sectionFields
        );

        $fields[]['form'] = array(
            'legend' => array(
                'title' => $this->l('Product page configuration'),
                'icon' => 'icon-cogs',
            ),
            'input' => $sectionFields,
            'submit' => array(
                'title' => $this->l('Save'),
                'type' => 'submit',
                'name' => 'submitwhatsappchatModule',
            ),
        );

        $positions = array(
            array(
                'id' => 'SHOPPINGCART',
                'name' => 'shopping cart'
            ),
            array(
                'id' => 'SHOPPINGCARTEXTRA',
                'name' => 'shopping cart extra'
            ),
            array(
                'id' => 'PAYMENTTOP',
                'name' => 'payment top'
            ),
            array(
                'id' => 'BEFORECARRIER',
                'name' => 'before carrier'
            ),
            array(
                'id' => 'ORDERCONFIRMATION',
                'name' => 'order confirmation'
            ),
        );

        $sectionFields = array();
        foreach ($positions as $position) {
            $sectionFields = array_merge($sectionFields, array(
                array(
                    'type'          => 'html',
                    'html_content'  => '<hr>',
                ),
                array(
                    'type' => 'radio',
                    'label' => $this->l('Display block in ').$position['name'].'?',
                    'name' => 'WA_'.$position['id'].'_ENABLE',
                    'is_bool' => true,
                    'class' => 't',
                    'values' => array(
                        array(
                            'id' => 'WA_'.$position['id'].'_ENABLE_on',
                            'value' => 1,
                            'label' => $this->l('Yes')),
                        array(
                            'id' => 'WA_'.$position['id'].'_ENABLE_off',
                            'value' => 0,
                            'label' => $this->l('No')),
                    ),
                ),
                array(
                    'type'  => 'text',
                    'label' => $this->l('Block message'),
                    'name'  => 'WA_'.$position['id'].'_MESSAGE',
                    'lang'  => true,
                    'col'   => 4,
                    'class' => 't',
                ),
                array(
                    'type' => 'select',
                    'label' => $this->l('Position'),
                    'name' => 'WA_'.$position['id'].'_POSITION',
                    'class' => 't',
                    'options' => array(
                        'query' => array(
                            array(
                                'id' => 'left',
                                'name' => $this->l('Left')
                            ),
                            array(
                                'id' => 'center',
                                'name' => $this->l('Center'),
                            ),
                            array(
                                'id' => 'right',
                                'name' => $this->l('Right'),
                            )
                        ),
                        'id' => 'id',
                        'name' => 'name'
                    ),
                ),
                array(
                    'type' => 'radio',
                    'label' => $this->l('Open chat in a new window?'),
                    'name' => 'WA_'.$position['id'].'_CHAT',
                    'is_bool' => true,
                    'class' => 't',
                    'values' => array(
                        array(
                            'id' => 'WA_'.$position['id'].'_CHAT_on',
                            'value' => 1,
                            'label' => $this->l('Yes')),
                        array(
                            'id' => 'WA_'.$position['id'].'_CHAT_off',
                            'value' => 0,
                            'label' => $this->l('No')),
                    ),
                )
            ));
        }

        $sectionFields = array_merge(
            array(array(
                'type'  => 'html',
                'html_content'  => '<div style="text-align:center; font-weight:bold"><a target="_blank" href="'.$this->_path.'views/img/shoppingcart_14.png">'.$this->l('View hook position 🔗').'</a></div><br />',
            )),
            $sectionFields
        );

        $fields[]['form'] = array(
            'legend' => array(
                'title' => $this->l('Checkout/Order configuration'),
                'icon' => 'icon-cogs',
            ),
            'input' => $sectionFields,
            'submit' => array(
                'title' => $this->l('Save'),
                'type' => 'submit',
                'name' => 'submitwhatsappchatModule',
            ),
        );

        $positions = array(
            array(
                'id' => 'CUSTOMERACCOUNT',
                'name' => 'customer account'
            ),
            array(
                'id' => 'MYACCOUNTBLOCK',
                'name' => 'my account'
            ),
        );

        $sectionFields = array();
        foreach ($positions as $position) {
            $sectionFields = array_merge($sectionFields, array(
                array(
                    'type'          => 'html',
                    'html_content'  => '<hr>',
                ),
                array(
                    'type' => 'radio',
                    'label' => $this->l('Display block in ').$position['name'].'?',
                    'name' => 'WA_'.$position['id'].'_ENABLE',
                    'is_bool' => true,
                    'class' => 't',
                    'values' => array(
                        array(
                            'id' => 'WA_'.$position['id'].'_ENABLE_on',
                            'value' => 1,
                            'label' => $this->l('Yes')),
                        array(
                            'id' => 'WA_'.$position['id'].'_ENABLE_off',
                            'value' => 0,
                            'label' => $this->l('No')),
                    ),
                ),
                array(
                    'type'  => 'text',
                    'label' => $this->l('Block message'),
                    'name'  => 'WA_'.$position['id'].'_MESSAGE',
                    'lang'  => true,
                    'col'   => 4,
                    'class' => 't',
                ),
                array(
                    'type' => 'select',
                    'label' => $this->l('Position'),
                    'name' => 'WA_'.$position['id'].'_POSITION',
                    'class' => 't',
                    'options' => array(
                        'query' => array(
                            array(
                                'id' => 'left',
                                'name' => $this->l('Left')
                            ),
                            array(
                                'id' => 'center',
                                'name' => $this->l('Center'),
                            ),
                            array(
                                'id' => 'right',
                                'name' => $this->l('Right'),
                            )
                        ),
                        'id' => 'id',
                        'name' => 'name'
                    ),
                ),
                array(
                    'type' => 'radio',
                    'label' => $this->l('Open chat in a new window?'),
                    'name' => 'WA_'.$position['id'].'_CHAT',
                    'is_bool' => true,
                    'class' => 't',
                    'values' => array(
                        array(
                            'id' => 'WA_'.$position['id'].'_CHAT_on',
                            'value' => 1,
                            'label' => $this->l('Yes')),
                        array(
                            'id' => 'WA_'.$position['id'].'_CHAT_off',
                            'value' => 0,
                            'label' => $this->l('No')),
                    ),
                )
            ));
        }

        $sectionFields = array_merge(
            array(array(
                'type'  => 'html',
                'html_content'  => '<div style="text-align:center; font-weight:bold"><a target="_blank" href="'.$this->_path.'views/img/shoppingcart_14.png">'.$this->l('View hook position 🔗').'</a></div><br />',
            )),
            $sectionFields
        );

        $fields[]['form'] = array(
            'legend' => array(
                'title' => $this->l('My account configuration'),
                'icon' => 'icon-cogs',
            ),
            'input' => $sectionFields,
            'submit' => array(
                'title' => $this->l('Save'),
                'type' => 'submit',
                'name' => 'submitwhatsappchatModule',
            ),
        );

        return $fields;
    }

    protected function getConfigFormValues()
    {
        $fields = array();

        $fields['WA_CHAT_MOBILE'] = Tools::getValue(
            'WA_CHAT_MOBILE',
            Configuration::get('WA_CHAT_MOBILE')
        );

        $fields['WA_FONT_AWESOME'] = Tools::getValue(
            'WA_FONT_AWESOME',
            Configuration::get('WA_FONT_AWESOME')
        );

        $positions = array('BADGE', 'HEADER', 'FOOTER', 'LEFTCOLUMN', 'RIGHTCOLUMN', 'TOP', 'HOME', 'EXTRALEFT', 'EXTRARIGHT', 'PRODUCTACTIONS', 'PRODUCTFOOTER', 'SHOPPINGCART', 'SHOPPINGCARTEXTRA', 'PAYMENTTOP', 'BEFORECARRIER', 'ORDERCONFIRMATION', 'CUSTOMERACCOUNT', 'MYACCOUNTBLOCK', 'TOPWIDTH', 'BOTTOMWIDTH');

        foreach ($positions as $position) {
            $fields['WA_'.$position.'_ENABLE'] = Tools::getValue(
                'WA_'.$position.'_ENABLE',
                Configuration::get('WA_'.$position.'_ENABLE')
            );

            $fields['WA_'.$position.'_POSITION'] = Tools::getValue(
                'WA_'.$position.'_POSITION',
                Configuration::get('WA_'.$position.'_POSITION')
            );

            $fields['WA_'.$position.'_CHAT'] = Tools::getValue(
                'WA_'.$position.'_CHAT',
                Configuration::get('WA_'.$position.'_CHAT')
            );
        }

        $languages = Language::getLanguages(false);
        foreach ($languages as $lang) {
            $fields['WA_CHAT_MESSAGE'][$lang['id_lang']] = Tools::getValue(
                'WA_CHAT_MESSAGE_'.$lang['id_lang'],
                Configuration::get('WA_CHAT_MESSAGE', $lang['id_lang'])
            );

            foreach ($positions as $position) {
                $fields['WA_'.$position.'_MESSAGE'][$lang['id_lang']] = Tools::getValue(
                    'WA_'.$position.'_MESSAGE_'.$lang['id_lang'],
                    Configuration::get('WA_'.$position.'_MESSAGE', $lang['id_lang'])
                );
            }
        }

        return $fields;
    }

    public function getWarnings($getAll = true, $install = false)
    {
        $warning = array();

        if (version_compare(_PS_VERSION_, '1.6', '>=')) {
            if (Configuration::get('PS_DISABLE_NON_NATIVE_MODULE')) {
                $warning[] = $this->l('You have to enable non PrestaShop modules at ADVANCED PARAMETERS - PERFORMANCE');
            }
        }

        if (version_compare(_PS_VERSION_, '1.5', '<')) {
            if (!(Configuration::get('WA_CHAT_MOBILE'))) {
                $warning[] = $this->l('You have to set your mobile phone number.');
            }
        } else {
            if (!$install) {
                $whatsappchatblock = new WhatsappChatBlock();
                $chats = $whatsappchatblock->getWhatsappChats(true);
                if (count($chats) == 0) {
                    $warning[] = $this->l('There are no WhatsApp chats configured / enabled.');
                }
            }
        }

        if (count($warning) && !$getAll) {
            return $warning[0];
        }

        return $warning;
    }

    public function getAvailableHooks()
    {
        return array(
            /* All pages */
            array(
                'id' => 'badge',
                'name' => $this->l('Badge - All pages')
            ),
            array(
                'id' => 'hookDisplayBanner',
                'name' => $this->l('Banner (displayBanner) - All pages')
            ),
            array(
                'id' => 'top',
                'name' => $this->l('Top (displayTop) - All pages')
            ),
            array(
                'id' => 'header',
                'name' => $this->l('Header (displayHeader) - All pages')
            ),
            array(
                'id' => 'footer',
                'name' => $this->l('Footer (displayFooter) - All pages')
            ),
            array(
                'id' => 'leftcolumn',
                'name' => $this->l('Left column (displayLeftColumn) - All pages')
            ),
            array(
                'id' => 'rightcolumn',
                'name' => $this->l('Right column (displayRightColumn) - All pages')
            ),
            array(
                'id' => 'hookDisplayTopColumn',
                'name' => $this->l('Top column (displayTopColumn) - All pages')
            ),
            array(
                'id' => 'hookDisplayNav',
                'name' => $this->l('Navigation (displayNav) - All pages')
            ),
            array(
                'id' => 'topWidth',
                'name' => $this->l('Top of page 100% width - All pages')
            ),
            array(
                'id' => 'bottomWidth',
                'name' => $this->l('Sticked at bottom of page 100% width  - All pages')
            ),

            /* Homepage */
            array(
                'id' => 'home',
                'name' => $this->l('Home (displayHome) - Homepage')
            ),

            /* Product page */
            array(
                'id' => 'hookDisplayProductButtons',
                'name' => $this->l('Product page actions (displayProductButtons) - Product page')
            ),
            array(
                'id' => 'hookDisplayLeftColumnProduct',
                'name' => $this->l('Left column (displayLeftColumnProduct) - Product page')
            ),
            array(
                'id' => 'hookDisplayRightColumnProduct',
                'name' => $this->l('Right column (displayRightColumnProduct) - Product page')
            ),
            array(
                'id' => 'hookDisplayFooterProduct',
                'name' => $this->l('Product footer (displayFooterProduct) - Product page')
            ),

            /* Checkout */
            array(
                'id' => 'shoppingcart',
                'name' => $this->l('Shopping cart (displayShoppingCart) - Checkout')
            ),
            array(
                'id' => 'shoppingcartextra',
                'name' => $this->l('Shopping cart extra (displayShoppingCartExtra) - Checkout')
            ),
            array(
                'id' => 'paymenttop',
                'name' => $this->l('Top of payment page (displayPaymentTop) - Checkout')
            ),
            array(
                'id' => 'beforecarrier',
                'name' => $this->l('Before carriers list (displayBeforeCarrier) - Checkout')
            ),

            /* Order */
            array(
                'id' => 'orderconfirmation',
                'name' => $this->l('Order confirmation (displayOrderConfirmation) - Order')
            ),

            /* Customer account */
            array(
                'id' => 'customeraccount',
                'name' => $this->l('Customer account (displayCustomerAccount) - Customer account')
            ),
            array(
                'id' => 'hookDisplayCustomerAccountForm',
                'name' => $this->l('Customer account creation form (displayCustomerAccountForm) - Customer account')
            ),
            array(
                'id' => 'hookDisplayCustomerAccountFormTop',
                'name' => $this->l('Above the customer account creation form (displayCustomerAccountFormTop) - Customer account')
            ),
            array(
                'id' => 'hookDisplayCustomerIdentityForm',
                'name' => $this->l('Customer identity form (displayCustomerIdentityForm) - Customer account')
            ),
            array(
                'id' => 'myaccountblock',
                'name' => $this->l('My account footer block (displayMyAccountBlock) - Customer account')
            ),

            /* Maintenance */
            array(
                'id' => 'hookDisplayMaintenance',
                'name' => $this->l('Maintenance (displayMaintenance) - Maintenance page')
            ),
        );
    }

    private function currentPageURL()
    {
        if (isset($_SERVER["HTTPS"]) && $_SERVER["HTTPS"] == "on") {
            {$pageURL = "https://";}
        } else {
            {$pageURL = "http://";}
        }

        if ($_SERVER["SERVER_PORT"] != "80") {
            $pageURL .= $_SERVER["SERVER_NAME"].":".$_SERVER["SERVER_PORT"].$_SERVER["REQUEST_URI"];
        } else {
            $pageURL .= $_SERVER["SERVER_NAME"].$_SERVER["REQUEST_URI"];
        }

        return $pageURL;
    }

    public function formatMobilePhoneForWhatsapp($phone, $id_country = false)
    {
        $phone = preg_replace('/[^0-9]+/', '', $phone);
        if (!$id_country) {
            return $phone;
        } else {
            $country = new Country((int)$id_country);
            if (Tools::substr($phone, 0, Tools::strlen($country->call_prefix)) == $country->call_prefix) {
                return $phone;
            } else {
                return $country->call_prefix.$phone;
            }
        }
    }

    public function getWhatsappUrl($phone = false, $text = false, $chat_group = '')
    {
	    $mobile = new Mobile_Detect();
        if ($mobile->isMobile() || $mobile->isTablet()) {
            $url = 'https://api.whatsapp.com/';
        } else {
            $url = 'https://web.whatsapp.com/';
        }
        if ($chat_group != '') {
            return 'https://chat.whatsapp.com/'.$chat_group;
        }
        return $url.'send?l='.$this->context->language->iso_code.($phone ? '&phone='.$phone : '').($text ? '&text='.$text : '');
    }

    public function getCustomerGroups($id_lang, $withall = true)
    {
        if ($withall) {
            $groups = array(array('id_group' => 'all', 'name' => $this->l('-- All --')));
        } else {
            $groups = array(array('id_group' => '0', 'name' => '-'));
        }
        return array_merge($groups, Group::getGroups($id_lang, true));
    }
}
